/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: ICollection.java,v 1.5 2004/05/27 11:33:52 jre Exp $
 */

package com.sapportals.wcm.repository;

import com.sapportals.wcm.repository.enum.LinkType;
import com.sapportals.wcm.repository.enum.OrderType;
import com.sapportals.wcm.util.content.IContent;
import com.sapportals.wcm.util.uri.URL;

/**
 * A collection is a resource ({@link IResource}) that contains other resources
 * and collections following the composite pattern. <p>
 *
 * Copyright (c) SAP AG 2001-2004
 * @author m.breitenfelder@sap.com
 * @version $Revision: 1.5 $
 */
public interface ICollection extends IResource {

  /**
   * Returns a list of all resources in the collection
   *
   * @return A list of resources
   * @exception ResourceException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResourceList getChildren()
    throws ResourceException, AccessDeniedException;

  /**
   * Returns a filtered list of the resources in this collection.
   *
   * @param visible true: only visible resource (system property "hidden" is
   *      false), false: only hidden resources
   * @param collections true: only collections, false: only resources which are
   *      not collections
   * @param links true: only links, false: only non-link resources
   * @return A list of resources
   * @exception ResourceException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResourceList getChildren(boolean visible, boolean collections, boolean links)
    throws ResourceException, AccessDeniedException;

  /**
   * Returns the number of children in the collection
   *
   * @param visible true: only visible resource (system property "hidden" is
   *      false), false: only hidden resources
   * @param collections true: only collections, false: only resources which are
   *      not collections
   * @param links true: only links, false: only non-link resources
   * @return childrenCount
   * @exception ResourceException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public int getChildrenCount(boolean visible, boolean collections, boolean links)
    throws ResourceException, AccessDeniedException;

  /**
   * Returns the children of this collection. The list of children can be filtered and
   * sorted by properties. A child resource is not included in the result if the user in the
   * context does not have read permission for the properties (this is true if at least one parameter is used (not <code>null</code>)
   * which will cause properties to be fetched: Collator, Selector or propertyPrefill).
   * @param childrenSelector Specifies the filter expression
   * @param sortBy Specifies the ordering of the returned list. Children which are collections are grouped together and are placed BEFORE the non-collections.
   * @param propertyPrefill Specifies the names of properties which should be fetched
   * for the children of the collection in addition to those needed for filtering and sorting. This "prefetching" can improve
   * performance of subsequent getProperty() calls on the child resources.
   * @exception ResourceException
   * @exception AccessDeniedException If the resource is locked, readonly or authorization failed
   */
  public IResourceList getChildren(Selector childrenSelector, Collator sortBy, IPropertyNameList propertyPrefill)  throws ResourceException, AccessDeniedException;

  /**
   * Returns the children of this collection. The list of children can be filtered by properties and permissions and
   * sorted by properties.  A child resource is not included in the result if the user in the
   * context does not have read permission for the properties (this is true if at least one parameter is used (not <code>null</code>)
   * which will cause properties to be fetched: Collator, Selector or propertyPrefill).
   * @param childrenSelector Specifies the filter expression
   * @param sortBy Specifies the ordering of the returned list.  Children which are collections are grouped together and are placed BEFORE the non-collections.
   * @param propertyPrefill Specifies the names of properties which should be fetched
   * for the children of the collection in addition to those needed for filtering and sorting. This "prefetching" can improve
   * performance of subsequent getProperty() calls on the child resources.
   * @param permissionNames An array of permission names (permission name constants are defined in IPermission).
   * Resources for which the user in the context does not have one of the secified permissions are removed from the result list.
   * @exception ResourceException
   * @exception AccessDeniedException If the resource is locked, readonly or authorization failed
   */
  public IResourceList getChildren(Selector childrenSelector, Collator sortBy, IPropertyNameList propertyPrefill, String[] permissionNames)  throws ResourceException, AccessDeniedException;

  /**
   * True if the collection has at least one child resource
   *
   * @exception ResourceException
   */
  public boolean hasChildren()
    throws ResourceException;


  /**
   * Returns the order type of a collection
   *
   * @return The order type
   * @exception ResourceException
   */
  public OrderType getOrderType()
    throws ResourceException;

  /**
   * Set the order type of a collection
   *
   * @param orderType orderType to be set
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public void setOrderType(OrderType orderType)
    throws ResourceException, NotSupportedException, AccessDeniedException;


  /**
   * Change the position of some or all resources in an ordered collection.
   * Positioning failures will be reported via one or more resource exception
   * (the RIDs will refer to the resource that couldn't be positioned).
   *
   * @param list A list that contains <code>IPositioning</code> instances that
   *      specify how the resources in the collection should be positioned.
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public void reorder(IReorderList list)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Creates a new sub collection in a collection.
   *
   * @param name The name of the new collection.
   * @param properties The intial properties to set for the new resource. The
   *      repository will try to set all properties in the map.
   * @return The new collection
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public ICollection createCollection(String name, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Creates a new sub collection in a collection that supports ordering.
   *
   * @param name The name of the new collection
   * @param position The position in the collection
   * @param properties The intial properties to set for the new resource. The
   *      repository will try to set all properties in the map.
   * @return The new collection
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public ICollection createCollection(String name, IPosition position, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Creates a new sub collection in a collection.
   *
   * @param name The name of the new collection.
   * @param properties The intial properties to set for the new resource. The
   *      repository will try to set all properties in the map.
   * @param ignorePropertyFailures Ignore exceptions if some or all properties
   *      could not be set/removed
   * @return The new collection
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public ICollection createCollection(String name, IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Creates a new sub collection in a collection that supports ordering.
   *
   * @param name The name of the new collection
   * @param position The position in the collection
   * @param properties The intial properties to set for the new resource. The
   *      repository will try to set all properties in the map.
   * @param ignorePropertyFailures Ignore exception if some or all properties
   *      could not be set/removed
   * @return The new collection
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public ICollection createCollection(String name, IPosition position, IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Creates a new resource in a collection.
   *
   * @param name The name of the resource
   * @param properties The intial properties to set for the new resource. The
   *      repository will try to set all properties in the map.
   * @param content The new content
   * @return The new resource
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResource createResource(String name, IPropertyMap properties, IContent content)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Creates a new resource in a collection that supports ordering.
   *
   * @param name The name of the resource
   * @param position Ths positioning information
   * @param properties The intial properties to set for the new resource. The
   *      repository will try to set all properties in the map.
   * @param content The new content
   * @return The new resource
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResource createResource(String name, IPosition position, IPropertyMap properties,
    IContent content)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Creates a new resource in a collection.
   *
   * @param name The name of the resource
   * @param properties The intial properties to set for the new resource. The
   *      repository will try to set all properties in the map.
   * @param content The new content
   * @param ignorePropertyFailures Ignore exception if some or all properties
   *      could not be set/removed
   * @return The new resource
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResource createResource(String name, IPropertyMap properties, IContent content, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Creates a new resource in a collection that supports ordering.
   *
   * @param name The name of the resource
   * @param position The positioning information
   * @param properties The intial properties to set for the new resource. The
   *      repository will try to set all properties in the map.
   * @param content The new content
   * @param ignorePropertyFailures
   * @return The new resource
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResource createResource(String name, IPosition position, IPropertyMap properties,
    IContent content, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Creates a empty resource that is initially locked (lock null resource)
   *
   * @param name The name of the resource
   * @param lockProperties The properties of the lock
   * @return LockInfo that contains the lock token
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public ILockInfo createResource(String name, ILockProperties lockProperties)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Creates a new link in a collection.
   *
   * @param name The name of the resource
   * @param targetURL The link target
   * @param linkType The link type
   * @param properties The intial properties to set for the new resource. The
   *      repository will try to set all properties in the map.
   * @return The new resource
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResource createLink(String name, URL targetURL, LinkType linkType,
    IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Creates a new link in a collection that supports ordering.
   *
   * @param name The name of the resource
   * @param targetURL The link target
   * @param linkType The link type
   * @param position The positioning information
   * @param properties The intial properties to set for the new resource. The
   *      repository will try to set all properties in the map.
   * @return The new resource
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResource createLink(String name, URL targetURL, LinkType linkType,
    IPosition position, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Creates a new link in a collection.
   *
   * @param name The name of the resource
   * @param targetURL The link target
   * @param linkType The link type
   * @param properties The intial properties to set for the new resource. The
   *      repository will try to set all properties in the map.
   * @param ignorePropertyFailures
   * @return The new resource
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResource createLink(String name, URL targetURL, LinkType linkType,
    IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Creates a new link in a collection that supports ordering.
   *
   * @param name The name of the resource
   * @param targetURL The link target
   * @param linkType The link type
   * @param position The positioning information
   * @param properties The intial properties to set for the new resource. The
   *      repository will try to set all properties in the map.
   * @param ignorePropertyFailures
   * @return The new resource
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResource createLink(String name, URL targetURL, LinkType linkType,
    IPosition position, IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException;
}

