/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import com.sapportals.wcm.util.content.ContentException;
import com.sapportals.wcm.util.content.IContent;

import java.io.IOException;
import java.io.InputStream;
import java.util.Date;

/**
 * Access to the content data of a resource. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author m.breitenfelder@sapportals.com
 * @author julian.reschke@greenbytes.de
 * @version $Revision: 1.20 $
 */

public class Content implements IContent, IMutableEntityMetadata {

  private final InputStream inputStream;
  private final String mimeType;
  private final long contentLength;
  private final String encoding;
  private final String language;

  private String entityTag = null;
  private Date lastModified = null;
  private boolean metadataWasUpdated = false;

  /**
   * Constructs Content object, for use when media subtype and encoding are
   * available as individual values.
   *
   * @param inputStream input stream
   * @param mediaType media type of content (trailing parameters are ignored)
   * @param contentLength length of content (use -1 when unknown)
   * @param encoding character encoding (use <code>null</code> when unknown)
   */
  public Content(InputStream inputStream, String mediaType, long contentLength, String encoding) {
    this.inputStream = inputStream;
    this.mimeType = getMimeType(mediaType);
    this.contentLength = contentLength;
    this.encoding = encoding;
    this.language = null;
  }

  /**
   * Constructs Content object, for use when media subtype and encoding are
   * available as individual values.
   *
   * @param inputStream input stream
   * @param mediaType media type of content (trailing parameters are ignored)
   * @param contentLength length of content (use -1 when unknown)
   * @param encoding character encoding (use <code>null</code> when unknown)
   * @param language content language (use <code>null</code> when unknown)
   */
  public Content(InputStream inputStream, String mediaType, long contentLength, String encoding, String language) {
    this.inputStream = inputStream;
    this.mimeType = getMimeType(mediaType);
    this.contentLength = contentLength;
    this.encoding = encoding == null ? getEncoding(mediaType) : encoding;
    this.language = language;
  }

  /**
   * Constructs Content object, for use when encoding is not known separately
   * but may be present in media type string.
   *
   * @param inputStream input stream
   * @param mediaType media type of content (trailing parameters except encoding
   *      information is ignored)
   * @param contentLength length of content (use -1 when unknown)
   */
  public Content(InputStream inputStream, String mediaType, long contentLength) {
    this.inputStream = inputStream;
    this.mimeType = getMimeType(mediaType);
    this.contentLength = contentLength;
    this.encoding = getEncoding(mediaType);
    this.language = null;
  }

  public InputStream getInputStream()
    throws ContentException {
    return this.inputStream;
  }

  public long getContentLength()
    throws ContentException {
    return this.contentLength;
  }

  public String getContentType()
    throws ContentException {
    return this.mimeType;
  }

  public String getEncoding()
    throws ContentException {
    return this.encoding;
  }

  public String getEntityTag()
    throws ResourceException {
    return this.entityTag;
  }

  public void setEntityTag(String entityTag)
    throws ResourceException {
    this.metadataWasUpdated = true;
    this.entityTag = entityTag;
  }

  public Date getExpiryDate()
    throws ResourceException {
    return null;
  }

  public String getLanguage()
    throws ResourceException {
    return this.language;
  }

  public Date getLastModified()
    throws ResourceException {
    return this.lastModified;
  }

  public void setLastModified(Date lastModified)
    throws ResourceException {
    this.metadataWasUpdated = true;
    this.lastModified = lastModified;
  }

  public boolean wasUpdated() {
    return this.metadataWasUpdated;
  }

  public void close() {
    try {
      this.inputStream.close();
    }
    catch (IOException ex) {
      //$JL-EXC$ 
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(ex.getMessage());
    }
  }


  // private

  private String getMimeType(String contentType) {

    if (contentType == null) {
      return null;
    }
    else {
      int pos = contentType.indexOf(";");
      if (pos < 0) {
        return contentType.trim();
      }
      else {
        return contentType.substring(0, pos).trim();
      }
    }
  }

  private String getEncoding(String contentType) {

    if (contentType == null) {
      return null;
    }
    else {
      int pos = contentType.indexOf(";");
      if (pos < 0) {
        return null;
      }
      else {
        String modifiers = contentType.substring(pos + 1);
        java.util.StringTokenizer st = new java.util.StringTokenizer(modifiers, ";");

        while (st.hasMoreTokens()) {

          String tok = st.nextToken().trim();

          if (tok.toLowerCase().startsWith("charset=")) {

            String val = tok.substring("charset=".length());
            int from = 0;
            int to = val.length();

            if (val.charAt(0) == '"' || val.charAt(0) == '\'') {
              from += 1;
            }

            if (val.charAt(to - 1) == '"' || val.charAt(to - 1) == '\'') {
              to -= 1;
            }

            return val.substring(from, to);
          }
        }

        return null;
      }
    }
  }
}
