/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: IFrameworkTransaction.java,v 1.1 2005/01/21 11:49:20 mbo Exp $
 */
package com.sapportals.wcm;

import java.sql.Connection;

import com.sapportals.wcm.repository.ResourceException;

/**
 * Interface for framework wide transactions. 
 * <p>
 * Copyright (c) SAP AG 2004
 * @author martin.boettcher@greenbytes.de
 */
public interface IFrameworkTransaction {
    
  /**
   * @return the connection to be used for all persistence calls
   */
  public Connection getConnection();   
  
  /**
   * Stores all kind of objects in this framework transaction. 
   * Can be retrieved via {@link #getTransactionObject}
   */
  public void addTransactionObject(Object key, Object o);

  /**
   * Returns a previous added object.
   * @param key the lookup key
   * @return null or the previously added object
   * @see #addTransactionObject(Object, Object)
   */
  public Object getTransactionObject(Object key);
  
  /**
   * Adds an {@link ITransactionEndsListener} which is called either when the transaction is committed or rollbacked.
   * @param listener
   */
  public void addEndListener(ITransactionEndsListener listener) throws ResourceException;
  
  /**
   * Tells if a specific listener is already registered with this transaction
   * @param listener
   */
  public boolean isEndListenerRegistered(ITransactionEndsListener listener);
  
  /**
   * Returns a specific listener if it is already registered with this transaction
   * @param key 
   */
  public ITransactionEndsListener getEndListener(String key);
  
  /**
   * Has be called when a a transaction shall be committed. All notifiers will be called and the connection closed.
   * If transaction can not be committed an implicit {@link #rollback()} will be called and false returned.
   * 
   * @return true if commit was performed, false otherwise 
   */
  public boolean commit();
  
  /**
   * Has be called when a a transaction shall be rollbacked. All notifiers will be called and the connection closed. 
   * @see #willBeRollbacked() 
   */
  public void rollback();
  
  /**
   * If called a later commit will always return with false and a rollback will be performed.
   * @see #willBeRollbacked() 
   */
  public void setVeto();
  
  /**
   * Check whether it is necessary to perform further actions with this open transaction. 
   * @return true if a {@link #setVeto} was called.
   */
  public boolean willBeRollbacked();
  
}
