/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.service.pipeline.producer;
import com.sap.tc.logging.Location;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.repository.ResourceFactory;
import com.sapportals.wcm.service.IServiceTypesConst;
import com.sapportals.wcm.service.pipeline.IPipelineService;
import com.sapportals.wcm.service.pipeline.RequestData;
import com.sapportals.wcm.service.pipeline.tools.IXmlParser;
import com.sapportals.wcm.util.logging.LoggingFormatter;

//import org.apache.regexp.*;
//ds import com.sapportals.wcm.util.regex.re.RE;
//ds import com.sapportals.wcm.util.regex.re.RESyntaxException;
import com.sapportals.wcm.util.regex.*;

import java.io.*;
import java.lang.reflect.Method;
import java.util.Properties;

import org.w3c.dom.*;
import org.xml.sax.*;

/**
 * Abstract base class for WCM Prodcuer. <p>
 *
 * <!--COPYRIGHT TAG:-->Copyright 2004 SAP AG<!-- -->
 *
 * @author roland.preussmann@sapportals.com
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/service/pipeline/api/com/sapportals/wcm/service/pipeline/producer/AbstractProducer.java#7
 *      $
 * @see IProducer
 */
public abstract class AbstractProducer implements IProducer {

  private final static String CONF_XML_MIME = "xml_mime";
  private final static String CONF_XML_EXT = "xml_ext";
  private final static String CONF_UNKNOWN_MIME = "unknown_mime";
  private final static String DEF_XML_MIME = ".*/xml";
  private final static String DEF_XML_EXT = "xml|xsl|xpdf|xsvg|xwml|xvrml";
  private final static String DEF_UNKNOWN_MIME = "text/plain|application/octet-stream";
  private final static String DUMMY_XML = "<?xml version=\"1.0\"?><DUMMY></DUMMY>";

  private static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.service.pipeline.producer.AbstractProducer.class);

  private static boolean m_initialize = false;
  private static IXmlParser m_parser = null;
  /*
   * ds RE
   */
  private static Pattern m_xml_mime;
  /*
   * ds RE
   */
  private static Pattern m_xml_ext;
  /*
   * ds RE
   */
  private static Pattern m_unknown_mime;

  /**
   * Construct object of class AbstractProducer.
   */
  public AbstractProducer() { }

  /**
   * @param inputStream
   * @param mime
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   * @todo: Description of the incoming method parameter
   * @todo: Description of the outgoing return value
   */
  protected RequestData createRequestData(InputStream inputStream, String mime)
    throws WcmException {
    Initialize();

    Document document = null;
    InputStream stream = null;
    String encoding = null;
    boolean isXML = isXML(mime, getPath());
    boolean isStream = !isXML && isStream(mime, getPath());
    try {
      // try
      InputSource input = new InputSource(inputStream);
      //input.setSystemId(getPath());
      if (!isStream) {
        document = m_parser.parse(input);
      }
    }
    catch (Exception e) {
      if (isXML) {
        throw (e instanceof WcmException) ? (WcmException)e : new WcmException(e);
      }
      isStream = true;
    }

    if (isStream) {
      stream = inputStream;
      document = createEmptyDom();
    }
    else {
      // get the encoding of the XML
      // at the moment there's no 'offical' way to get the encoding out of the
      // DOM level 2. But some DOM implementation provides an addional API for this.
      // To avoid dependencies to these implenemtaion the reflecation API is
      // use.
      //
      // Xerces 1.3.x: document.getEncoding()
      // InQMy 2.0.x: document.getEncoding()
      try {
        Method method = document.getClass().getMethod("getEncoding", null);
        if (method != null) {
          encoding = (String)method.invoke(document, null);
        }
      }
      catch (Exception e) {
        log.infoT("createRequestData(135)", "no method: getEncoding() - using default encoding");
      }
    }
    return new RequestData(document, stream, mime, encoding);
  }

  /**
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   * @todo: Description of the Method.
   * @todo: Description of the outgoing return value
   */
  protected Document createEmptyDom()
    throws WcmException {
    return createDOM(DUMMY_XML);
  }

  /**
   * @param xml
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   * @todo: Description of the outgoing return value
   */
  protected Document createDOM(String xml)
    throws WcmException {
    Document document = null;
    StringReader is = new StringReader(xml);
    try {
      document = m_parser.parse(new InputSource(is));
    }
    catch (SAXException e) {
      throw new WcmException(e);
    }
    catch (IOException e) {
      throw new WcmException(e);
    }
    return document;
  }

  /**
   * Get the XML attribute of the AbstractProducer class.
   *
   * @param mime
   * @param urn
   * @return The XML value
   * @todo: Description of the incoming method parameter
   * @todo: Description of the incoming method parameter
   */
  private static boolean isXML(String mime, String urn) {
    /*
     * ds
     * boolean xml = (mime != null && m_xml_mime.match(mime));
     * if (!xml) xml = m_xml_ext.match(getExt(urn));
     * return xml;
     */
    return (mime != null && m_xml_mime.matcher(new CharSequenceImpl(mime)).matches())
       || m_xml_ext.matcher(new CharSequenceImpl(getExt(urn))).matches();
  }

  /**
   * Get the Stream attribute of the AbstractProducer class.
   *
   * @param mime
   * @param urn
   * @return The Stream value
   * @todo: Description of the incoming method parameter
   * @todo: Description of the incoming method parameter
   */
  private static boolean isStream(String mime, String urn) {
    /*
     * ds
     * boolean stream = (mime != null && ! m_unknown_mime.match(mime));
     * return stream;
     */
    return mime != null && !m_unknown_mime.matcher(new CharSequenceImpl(mime)).matches();
  }

  /**
   * Get the Ext attribute of the AbstractProducer class.
   *
   * @param urn
   * @return The Ext value
   * @todo: Description of the incoming method parameter
   */
  private static String getExt(String urn) {
    String ext = "";
    int index = urn.lastIndexOf(".");
    if (index > 0) {
      ext = urn.substring(index + 1, urn.length());
    }
    return ext;
  }

  /**
   * @exception WcmException Exception raised in failure situation
   * @todo: Description of the Method.
   */
  private static void Initialize()
    throws WcmException {
    if (m_initialize) {
      return;
    }

    // getXML parser tool
    IPipelineService pipeline = (IPipelineService)ResourceFactory.getInstance().getServiceFactory().getService(IServiceTypesConst.PIPELINE_SERVICE);
    m_parser = pipeline.getXmlParser();
    if (m_parser == null) {
      throw new WcmException("Parser not defined!");
    }

    // get config entries
    String xml_mime = null;
    String xml_ext = null;
    String unknown_mime = null;
    try {
      Properties props = pipeline.getConfiguration();
      xml_mime = props.getProperty(CONF_XML_MIME);
      xml_ext = props.getProperty(CONF_XML_EXT);
      unknown_mime = props.getProperty(CONF_UNKNOWN_MIME);
    }
    catch (Exception ex) {
      // ignore the exception. If the configuration entries are missing default
      // values are used.
      log.debugT(ex.getMessage());
    }

    // get xml mimes and extension
    try {
      if (xml_mime == null) {
        xml_mime = DEF_XML_MIME;
      }
      if (xml_ext == null) {
        xml_ext = DEF_XML_EXT;
      }
      if (unknown_mime == null) {
        unknown_mime = DEF_UNKNOWN_MIME;
      }

      m_xml_mime = new Pattern(xml_mime, Pattern.CASE_INSENSITIVE);
      m_xml_ext = new Pattern(xml_ext, Pattern.CASE_INSENSITIVE);
      m_unknown_mime = new Pattern(unknown_mime, Pattern.CASE_INSENSITIVE);
    }
    catch (
      PatternSyntaxException e) {
      log.errorT("Initialize(280)", e.toString());
      if (m_xml_mime == null) {
        m_xml_mime = new Pattern("");
      }
      if (m_xml_ext == null) {
        m_xml_ext = new Pattern("");
      }
      if (m_unknown_mime == null) {
        m_unknown_mime = new Pattern("");
      }
    }
    m_initialize = true;
  }
}
