/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.service.pipeline.processor;

import com.sapportals.wcm.*;
import com.sapportals.wcm.service.pipeline.IConst;
import com.sapportals.wcm.service.pipeline.RequestData;

import java.util.ArrayList;
import java.util.Dictionary;

import org.w3c.dom.*;

/**
 * Implementation of a default processor, which can be used to add processing
 * instructions to the xml. The DP is called by the pipeline controller every
 * time when the XML contains no further processing instruction. <p>
 *
 * <!--COPYRIGHT TAG:-->Copyright 2004 SAP AG<!-- -->
 *
 * @author roland.preussmann@sapportals.com
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/service/pipeline/api/com/sapportals/wcm/service/pipeline/processor/DefaultProcessor.java#4
 *      $
 */
public class DefaultProcessor extends WcmObject implements IProcessor, IConst {
  /**
   * TBD: Description of the class.
   */
  private class PIData {
    private String m_target;
    private String m_data;
    private Document m_dom;
    private int m_step;

    public PIData(int s, String t, String d) {
      m_step = s;
      m_target = t;
      m_data = d;
      m_dom = null;
    }

    public String getTarget() {
      return m_target;
    }

    public String getData() {
      return m_data;
    }

    public Document getDom() {
      return m_dom;
    }

    public void setDom(Document dom) {
      m_dom = dom;
    }

    public int getStep() {
      return m_step;
    }
  }

  private PIData m_formatter = null;
  private ArrayList m_processors;
  private int m_index;
  private int m_step;

  public DefaultProcessor() {
    m_index = 0;
    m_step = 0;
    m_processors = new ArrayList();
  }

  public RequestData process(RequestData data, Dictionary parameters)
    throws Exception {
    process(data.getDocument(), parameters);
    return data;
  }

  public Document process(Document document, Dictionary parameters)
    throws Exception {
    if (m_index < m_processors.size()) {
      boolean step = true;
      while (m_index < m_processors.size() && step) {
        PIData pid = (PIData)m_processors.get(m_index);
        ProcessingInstruction pi = createPI(document, pid);
        step = (m_step == pid.getStep());
        if (step) {
          insertPI(document, pi);
          m_index++;
        }
      }
      m_step++;
    }
    else if (m_index == m_processors.size() && m_formatter != null) {
      ProcessingInstruction pi = createPI(document, m_formatter);
      insertPI(document, pi);
      m_index++;
    }

    return document;
  }

  public void addProcessingInstruction(int step, String target, String data) {
    PIData pi = new PIData(step, target, data);
    m_processors.add(pi);
  }

  public void addProcessingInstruction(int step, String target, String href, String type) {
    String data = "";
    if (href != null && href.length() > 0) {
      data = "href=\"" + href + "\"";
    }
    if (type != null && type.length() > 0) {
      if (data.length() > 0) {
        data += " ";
      }
      data += "type=\"" + type + "\"";
    }
    addProcessingInstruction(step, target, data);
  }

  public void addWcmProcess(int step, String name) {
    addProcessingInstruction(step, WCM_PROCESS_PI, "", name);
  }

  public void addWcmProcess(int step, String name, Document dom) {
    addProcessingInstruction(step, WCM_PROCESS_PI, "", name);
    PIData pid = (PIData)m_processors.get(m_index);
    pid.setDom(dom);
  }

  public void addWcmFormat(String name) {
    String data = "type=\"" + name + "\"";
    m_formatter = new PIData(-1, WCM_FORMAT_PI, data);
  }

  /**
   * Creates a new ProcessingInstruction
   *
   * @param document TBD: Description of the incoming method parameter
   * @param pid TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  private ProcessingInstruction createPI(Document document, PIData pid) {
    return document.createProcessingInstruction(pid.getTarget(), pid.getData());
  }

  /**
   * insert a ProcessingInstruction before the first none PI node
   *
   * @param document TBD: Description of the incoming method parameter
   * @param pi TBD: Description of the incoming method parameter
   */
  private void insertPI(Document document, ProcessingInstruction pi) {

    NodeList nodelist = document.getChildNodes();
    Node node = null;
    int i = nodelist.getLength();
    for (int j = 0; j < i; j++) {
      node = nodelist.item(j);
      if (node.getNodeType() != Node.PROCESSING_INSTRUCTION_NODE) {
        break;
      }
    }
    document.insertBefore(pi, node);
  }
}

