/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.service.pipeline;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.service.IService;
import com.sapportals.wcm.service.pipeline.processor.IProcessor;
import com.sapportals.wcm.service.pipeline.producer.IProducer;
import com.sapportals.wcm.util.content.IContent;
import java.io.InputStream;

import java.util.Hashtable;

/**
 * The XSLT-Pipeline is a specialized version of the pipeline. <p>
 *
 * XSLT-Pipeline is used to transform XML into another form via XSL. The code
 * sample creates a XSLT pipeline and transforms a XML resource: <p>
 *
 * <pre>
 * IResource xml = ...
 * IResource xsl = ...
 * IXsltPipelineService xsltpl;
 * xsltpl = (IXsltPipelineService) ResourceFactory.getInstance().getServiceFactory().
 *           getService("xsltpipeline");
 * IContent content = xsltpl.handle(ProducerFromResource(xml),
 *                                  ProducerFromResource(xsl),
 *                                  null)
 * </pre> Copyright (c) SAP Portals Europe GmbH 2001
 *
 * @author roland.preussmann@sapportals.com
 * @version $Id:$
 */
public interface IXsltPipelineService extends IService {

  /**
   * WARNING: Deprecated Method (This version is error prone) !!! Use methods
   * which use mime type directly. Method sets the mime type.
   *
   * @param mime Mime type e.g. "text/html".
   */
  public void setFormat(String mime);

  /**
   * WARNING: Deprecated Method (This version is error prone) !!! Method gets
   * the mime type.
   *
   * @return MimeType
   */
  public String getFormat();

  /**
   * This method calls the XSLT processor. The return value contains the result
   * of the XSL transformation. For a better performance it is recommended to
   * use <code>ProducerFromResource</code> for the XSL source.
   *
   * @param xsl XSL Source. If <code>NULL</code> the XSL stylesheet must be
   *      defined in the XML.
   * @param properties Properties to pass to processor and formatter. If null
   *      then no parameters are passed. See {@link IPipelineService#handle(IProducer, IProcessor, Hashtable)}
   * @param xml TBD: Description of the incoming method parameter
   * @return result of the XSL transformation
   * @exception WcmException Exception raised in failure situation
   */
  public IContent handle(IProducer xml, IProducer xsl, Hashtable properties)
    throws WcmException;

  /**
   * This method calls the XSLT processor. The return value contains the result
   * of the XSL transformation. For a better performance it is recommended to
   * use <code>ProducerFromResource</code> for the XSL source.
   *
   * @param xml XML Source. See also {@link IPipelineService#handle(IProducer, IProcessor, Hashtable)}
   * @param xsl XSL Source. If <code>NULL</code> the XSL stylesheet must be
   *      defined in the XML.
   * @param properties Properties to pass to processor and formatter. If null
   *      then no parameters are passed. See {@link IPipelineService#handle(IProducer, IProcessor, Hashtable)}
   * @param mime Mime type for formatter. If <code>mime=null</code> then
   *      "text/html" is used
   * @return result of the XSL transformation
   * @exception WcmException Exception raised in failure situation
   */
  public IContent handle(IProducer xml, IProducer xsl, Hashtable properties, String mime)
    throws WcmException;

  /**
   * This method calls the XSLT processor. It should only be used for debugging.
   * The return value contains the result of the XSL transformation. Use methods
   * with the <code>Iproducer</code> interface instead.
   *
   * @param xml Inputstream to the XML.
   * @param xsl Inputstream to the XSL. If <code>NULL</code> the XSL stylesheet
   *      must be defined in the XML.
   * @param properties Properties to pass to processor and formatter. If null
   *      then no parameters are passed. See {@link IPipelineService#handle(IProducer, IProcessor, Hashtable)}
   * @return result of the XSL transformation
   * @exception WcmException Exception raised in failure situation
   */
  public IContent handle(InputStream xml, InputStream xsl, Hashtable properties)
    throws WcmException;

  /**
   * This method calls the XSLT processor. It should only be used for debugging.
   * The return value contains the result of the XSL transformation. Use methods
   * with the <code>Iproducer</code> interface instead.
   *
   * @param xml Inputstream of XML.
   * @param xsl Inputstream of XSL. If <code>NULL</code> the XSL stylesheet must
   *      be defined in the XML.
   * @param properties Properties to pass to processor and formatter. If null
   *      then no parameters are passed. See {@link IPipelineService#handle(IProducer, IProcessor, Hashtable)}
   * @param mime Mime type for formatter. If <code>mime=null</code> then
   *      "text/html" is used
   * @return result of the XSL transformation
   * @exception WcmException Exception raised in failure situation
   */
  public IContent handle(InputStream xml, InputStream xsl, Hashtable properties, String mime)
    throws WcmException;

  /**
   * This method calls the XSLT processor. The return value contains the result
   * of the XSL transformation.
   *
   * @param xml XML source
   * @param xslUrl XSL source
   * @param properties Properties to pass to processor and formatter. If null
   *      then no parameters are passed. See {@link IPipelineService#handle(IProducer, IProcessor, Hashtable)}
   * @return result of the XSL transformation
   * @exception WcmException Exception raised in failure situation
   */
  public IContent handle(IProducer xml, String xslUrl, Hashtable properties)
    throws WcmException;

  /**
   * This method calls the XSLT processor. The return value contains the result
   * of the XSL transformation.
   *
   * @param xml XML source
   * @param xslUrl XSL source
   * @param properties Properties to pass to processor and formatter. If null
   *      then no parameters are passed. See {@link IPipelineService#handle(IProducer, IProcessor, Hashtable)}
   * @param mime Mime type for formatter. If <code>mime=null</code> then
   *      "text/html" is used
   * @return result of the XSL transformation
   * @exception WcmException Exception raised in failure situation
   */
  public IContent handle(IProducer xml, String xslUrl, Hashtable properties, String mime)
    throws WcmException;

  /**
   * This method calls the XSLT processor. The return value contains the result
   * of the XSL transformation.
   *
   * @param xmlUrl XML source
   * @param xslUrl XSL source
   * @param properties (optional) Properties are passed to the XSLT processor.
   * @return result of the XSL transformation
   * @exception WcmException Exception raised in failure situation
   */
  public IContent handle(String xmlUrl, String xslUrl, Hashtable properties)
    throws WcmException;

  /**
   * This method calls the XSLT processor. The return value contains the result
   * of the XSL transformation.
   *
   * @param xmlUrl XML source
   * @param xslUrl XSL source
   * @param properties (optional) Properties are passed to the XSLT processor.
   * @param mime Mime type for formatter. If mime=null then "text/html" is used
   * @return result of the XSL transformation
   * @exception WcmException Exception raised in failure situation
   */
  public IContent handle(String xmlUrl, String xslUrl, Hashtable properties, String mime)
    throws WcmException;

}
