package com.inqmy.ats.system;

import java.util.Properties;
import java.net.ProtocolException;
import java.io.IOException;
import java.io.File;

import com.sap.sdm.api.remote.*;
import com.sap.sdm.api.remote.deployresults.*;
import com.sap.sdm.api.remote.deployresults.Success;
import com.sap.sdm.api.remote.deployresults.Warning;
import com.sap.sdm.api.remote.undeployresults.*;
import com.sap.sdm.api.remote.undeployresults.Error;
import com.inqmy.ats.SDMEnvironment;
import com.inqmy.ats.LogEnvironment;
import com.inqmy.ats.TestEnvironment;

/**
 *
 * @author Lalo Ivanov
 */
public class SDMEnvironmentImpl implements SDMEnvironment {

  private LogEnvironment logEnv = null;
  private TestEnvironment testEnv = null;
  private ClientSession session = null;
  private Client sdmClient = null;
  private HelperFactory factory = null;

  public void init(Properties p) throws Exception {
    logEnv =
      (LogEnvironment) EnvironmentFactory.getEnvironment(
        EnvironmentFactory.LOG);
    testEnv =
      (TestEnvironment) EnvironmentFactory.getEnvironment(
        EnvironmentFactory.TEST);
    int serverPort = Integer.parseInt( p.getProperty("SDMPort", "50018") );
    String serverHost = p.getProperty("SDMHost", "localhost");
    session = ClientSessionFactory.createRemoteClientSession(serverPort, serverHost , p.getProperty("SDMPassword" , "sdm"));
    sdmClient = session.getClient();
    factory = sdmClient.getHelperFactory();
    logEnv.log(session + " successfully created.");
  }

  public int[] deploy(String[] deployableFile) throws IOException, ProtocolException {
    // deploy at any case
    return deploy(deployableFile, UPDATE_ALL);
  }

  public int[] deploy(String[] deployableFiles, int versionRule) throws IOException, ProtocolException {
    try {
      DeployProcessor deployProcessor = sdmClient.getDeployProcessor();
      ComponentVersionHandlingRule versionHandlingRule =
        factory.createComponentVersionHandlingRule(versionRule);
      deployProcessor.setComponentVersionHandlingRule(versionHandlingRule);

      DeployItem[] items = new DeployItem[deployableFiles.length];
      for (int i = 0; i < deployableFiles.length; i++) {
        File f = new File(deployableFiles[i]);
        if(!f.exists()) {
          f = new File( new File( testEnv.getRootDir()), deployableFiles[i].replace(File.separatorChar, '/') );
        }
        items[i] = factory.createDeployItem(f);
      }
      deployProcessor.deploy(items);
      int[] returnCodes = logAndTransformDeployResult(deployableFiles, items);
      return returnCodes;
    } catch (RemoteException e) {
      logEnv.log(e);
      throw new ProtocolException(e.getMessage());
    }
  }

  private int[] logAndTransformDeployResult( String[] deployableFiles, DeployItem[] item) throws RemoteException {
    int codes[] = new int[item.length];
    for (int i = 0; i < item.length; i++) {
      DeployResult deployResult = item[i].getDeployResult();
      DeployResultType deployResultType = deployResult.getType();
      logEnv.log(
        "Deployment of "
          + deployableFiles[i]
          + " has result: " + deployResult.getResultText());

      if (deployResultType instanceof Aborted) {
        codes[i] = DEPLOY_ABORTED;
      } else if (deployResultType instanceof Admitted) {
        codes[i] = DEPLOY_ADMITTED;
      } else if (deployResultType instanceof AlreadyDeployed) {
        codes[i] = DEPLOY_ALREADY_DEPLOYED;
      } else if (deployResultType instanceof Executed) {
        codes[i] = DEPLOY_EXECUTED;
      } else if (deployResultType instanceof Initial) {
        codes[i] = DEPLOY_INITIAL;
      } else if (deployResultType instanceof com.sap.sdm.api.remote.deployresults.NotExecuted) {
        codes[i] = DEPLOY_NOT_EXECUTED;
      } else if (deployResultType instanceof PreconditionViolated) {
        codes[i] = DEPLOY_PRECONDITION_VIOLATED;
      } else if (deployResultType instanceof Rejected) {
        codes[i] = DEPLOY_REJECTED;
      } else if (deployResultType instanceof Success) {
        codes[i] = DEPLOY_SUCCESS;
      } else if (deployResultType instanceof Warning) {
        codes[i] = DEPLOY_WARNING;
      }
    }

    return codes;
  }

  public void checkSDMConfiguration() throws Exception {
    SDMConfig config = sdmClient.getSDMConfiguration();
    ////////////////////////////////////////////////////////////////////////////
    // Supported Server Types
    ////////////////////////////////////////////////////////////////////////////
    ServerType sTypes[] = config.getServerTypes();
    if (sTypes == null) {
      logEnv.log("Server doesn't support any server types");
    } else {
      logEnv.log("Supported server types by SDM Server: ");
      for (int i = 0; i < sTypes.length; i++) {
        logEnv.log(
          "Name: "
            + sTypes[i].getName()
            + ", Description: "
            + sTypes[i].getDescription());
      }
    }

    ////////////////////////////////////////////////////////////////////////////
    // Target Systems Info
    ////////////////////////////////////////////////////////////////////////////
    TargetSystemContainer tgContainer = config.getTargetSystemContainer();
    TargetSystem tgSystems[] = tgContainer.getTargetSystems();

    if (tgSystems == null) {
      logEnv.log("Server doesn't have any target systems");
    } else {
      FixedSizeParamContainer tgSystemConfig = null;
      for (int i = 0; i < tgSystems.length; i++) {
        logEnv.log(
          "Server type: "
            + tgSystems[i].getType().getName()
            + ", IsFullConfiguration: "
            + tgSystems[i].isConfigurationComplete());
        tgSystemConfig = tgSystems[i].getConfiguration();
        Param tgSystemConfigParams[] = tgSystemConfig.getParams();
        if (tgSystemConfigParams == null) {
          logEnv.log(" No parameters");
        } else {
          for (int j = 0; j < tgSystemConfigParams.length; j++) {
            String paramString =
              "Param name: " + tgSystemConfigParams[j].getName() + " , Value: ";
            int type = tgSystemConfigParams[j].getType().getTypeAsInt();
            switch (type) {
              case ParamTypes.BOOLEAN :
                paramString = paramString + tgSystemConfigParams[j].getValueBoolean();
                break;
              case ParamTypes.BYTE :
                paramString = paramString + tgSystemConfigParams[j].getValueByte();
                break;
              case ParamTypes.DOUBLE :
                paramString = paramString + tgSystemConfigParams[j].getValueDouble();
                break;
              case ParamTypes.FLOAT :
                paramString = paramString + tgSystemConfigParams[j].getValueFloat();
                break;
              case ParamTypes.INT :
                paramString = paramString + tgSystemConfigParams[j].getValueInt();
                break;
              case ParamTypes.LONG :
                paramString = paramString + tgSystemConfigParams[j].getValueLong();
                break;
              case ParamTypes.SHORT :
                paramString = paramString + tgSystemConfigParams[j].getValueShort();
                break;
              case ParamTypes.STRING :
                paramString = paramString + tgSystemConfigParams[j].getValueString();
                break;
              case ParamTypes.UNKNOWN :
                paramString = paramString + tgSystemConfigParams[j].getValueObject();
                break;
            }
            logEnv.log(paramString);
          } // for J
        } // else
      } // for I
    } // else

  }

  public int[] undeploy(String[][] componentNameAndVendor)
    throws Exception {
    try {
      UnDeployProcessor undeployProcessor = sdmClient.getUnDeployProcessor();
      UnDeployItem[] items = new UnDeployItem[componentNameAndVendor.length];
      for (int i=0; i<componentNameAndVendor.length; i++) {
        items[i] = factory.createUndeployItem(componentNameAndVendor[i][0], componentNameAndVendor[i][1]);
      }

      undeployProcessor.undeploy(items);
      return logAndTransformUndeployResult(componentNameAndVendor, items);
    } catch(RemoteException e) {
      logEnv.log(e);
      throw new ProtocolException(e.getMessage());
    }
  }

  private int[] logAndTransformUndeployResult(String[][] componentNameAndVendor, UnDeployItem[] items) throws Exception {
    int codes[] = new int[items.length];
    for (int i = 0; i < items.length; i++) {
      UnDeployResult undeployResult = items[i].getUnDeployResult();
      UnDeployResultType undeployResultType = undeployResult.getType();
      logEnv.log(
        "Undeployment of {"
          + componentNameAndVendor[i][0] + ", " + componentNameAndVendor[i][1]
          + " }has result: " + undeployResult.getResultText());

      if (undeployResultType instanceof Error) {
        codes[i] = UNDEPLOY_ERROR;
      } else if (undeployResultType instanceof com.sap.sdm.api.remote.undeployresults.NotExecuted) {
        codes[i] = UNDEPLOY_NOT_EXECUTED;
      } else if (undeployResultType instanceof Warning) {
        codes[i] = UNDEPLOY_WARNING;
      } else if (undeployResultType instanceof Success) {
        codes[i] = UNDEPLOY_SUCCESS;
      }
    }
    return codes;
  }

  public void close() throws Exception {
    session.closeSession();
  }
}

