package com.inqmy.ats;


import com.sap.sdm.api.remote.ComponentVersionHandlingRules;

import java.io.IOException;
import java.net.ProtocolException;

/**
 * This interface represents an environment for working with Software
 * Deployment Manager(SDM). SDM properties for connection to SDM server
 * (host, port) are taken from ATS default properties.  
 * 
 * @author Lalo Ivanov
 */
public interface SDMEnvironment extends Environment {

  /**
   * Deployment result constants
   */
  public static final int DEPLOY_SUCCESS = 0x1;
  public static final int DEPLOY_WARNING = 0x2;
  public static final int DEPLOY_ABORTED = 0x4;
  public static final int DEPLOY_EXECUTED = 0x7;
  public static final int DEPLOY_INITIAL = 0x8;
  public static final int DEPLOY_ADMITTED = 0x10;
  public static final int DEPLOY_PRECONDITION_VIOLATED = 0x20;
  public static final int DEPLOY_ALREADY_DEPLOYED = 0x40;
  public static final int DEPLOY_REJECTED = 0x60;
  public static final int DEPLOY_NOT_EXECUTED = 0x78;

  /**
   * Undeployment result constants
   */
  public static final int UNDEPLOY_SUCCESS = 0x1;
  public static final int UNDEPLOY_WARNING = 0x2;
  public static final int UNDEPLOY_NOT_EXECUTED = 0x4;
  public static final int UNDEPLOY_ERROR = 0x8;

  /**
   * Component version handling rules
   * Note: Error prone - the constants are compilation time linked with
   * the values in SDMClient.jar
   */
  public static final int UPDATE_ALL =
    ComponentVersionHandlingRules.UPDATE_ALL_VERSIONS;
  public static final int UPDATE_LOWER =
    ComponentVersionHandlingRules.UPDATE_LOWER_VERSIONS_ONLY;
  public static final int UPDATE_SAME_AND_LOWER =
    ComponentVersionHandlingRules.UPDATE_SAME_AND_LOWER_VERSIONS_ONLY;
  
  /**
   * Deploys an array of deployable files. Deployable files can be
   * of all types - all Engine types, also DB and FS deployables.
   * Method can be used directly since connection to SDM server is
   * already establshed. This method deploys components regardless
   * of the already deployed components' versions. 
   * 
   * @param deployableFile array with file names on the underlying FS
   * @return array of constants which are deploy results. Each one return
   *         code on i-th position corresponds to the deployment action
   *         executed on i-th <code>deployableFile</code>
   * @throws IOException thrown in case of file names not corresponding
   *                     to real files on the underlying FS
   * @throws ProtocolException thrown in case where an error occured in
   *                           the underying protocol used for communication
   *                           between SDM client and server
   */
  public int[] deploy(String[] deployableFile) 
                              throws IOException, Exception;


  /**
   * Deploys an array of deployable files. Deployable files can be
   * of all types - all Engine types, also DB and FS deployables.
   * Method can be used directly since connection to SDM server is
   * already establshed.
   * 
   * @param deployableFile array with file names on the underlying FS
   * @param versionRule the versioning rule that is to be applied, setting
   *                    this to unsiutable value could cause deployable file
   *                    not to be deployed
   * @return array of constants which are deploy results. Each one return
   *         code on i-th position corresponds to the deployment action
   *         executed on i-th <code>deployableFile</code>
   * @throws IOException thrown in case of file names not corresponding
   *                     to real files on the underlying FS
   * @throws ProtocolException thrown in case where an error occured in
   *                           the underying protocol used for communication
   *                           between SDM client and server
   */
  public int[] deploy(String[] deployableFile, int versionRule)
                              throws IOException, Exception;
  /**
   * Retrieves SDM server configuration properties and logs them.
   * 
   * @throws ProtocolException thrown in case where an error occured in
   *                           the underying protocol used for communication
   *                           between SDM client and server
   */
  public void checkSDMConfiguration() throws Exception;

  /**
   * Undeploys component.
   * 
   * @param componentNameAndVendor every row in this two-dimensional array
   *                               contains information about one component:
   *                               name of the component to be undeployed in first
   *                               column and vendor in second
   * @return array of constants which are undeploy results. Each one return
   *         code on i-th position corresponds to the undeployment action
   *         executed on i-th <code>componentNameAndVendor</code>
   * @throws ProtocolException thrown in case where an error occured in
   *                           the underying protocol used for communication
   *                           between SDM client and server
   */
  public int[] undeploy(String[][] componentNameAndVendor)
                              throws Exception ;

}