package com.inqmy.ats;

import com.sap.engine.services.deploy.container.FileUpdateInfo;

import java.util.*;

public interface DeployEnvironment extends Environment {


  /**
   * Deploys components found in the archive file to the specified container using given remote support
   * and applying properties necessary for deployment.
   *
   * @param archiveFile    the name of the archive that is being deployed.
   * @param containerName  the name of the container for which this archive is.
   * @param remoteSupport  the remote support that this components will have.
   * @param props          properties holding information necessary for the process of deployment;
   *                       they indicate J2EE Specification type, root lookup, container type.
   *
   * @return               array of names of the deployed components concatanated with
   *                       dash and String representation of their type.
   *
   * @exception   Exception  If some of the components could not be deployed
   *                               or errors occured while reading archive, for example
   *                               inproper xml format or missing classes.
   */
  public String[] deploy(String archiveFile, String containerName, String[] remoteSupport, Properties props) throws Exception;

  /**
   * Deploys all recognized components found in the archive file using specified remote support
   * and applying given properties necessary for deployment.If passed property is null then
   * default test properties are used. This method blocks calling thread until deployied
   * compenents are started.If timeout for starting expires then Exception is thrown.
   *
   * @param earFile           the file which is transfered on the local machine and points to the ear.
   *                          It has to represent relative file path from test root directory.
   * @param getRemoteSupports if true then gets a jar file containing all classes that client application
   *                          would need to run the specified application.
   * @param props             properties holding information necessary for the process of deployment;
   *                          they indicate J2EE Specification type, root lookup, container type.
   *
   * @return                  array of names of the deployed components concatanated with
   *                          dash and String representation of their type.
   *
   * @exception Exception     If some of the components could not be deployed
   *                          or errors occurred while reading archive, for example
   *                          inproper xml format of ear descriptor or component descriptors.
   */
  public String[] deployComponentsInEar(String earFile, boolean getRemoteSupports, Properties props) throws Exception;

  /**
   * Deploys component found in the archive file using specified remote support( placed in default test properties
   * or properties which test has passed to DeployEnvironemnt). This method blocks calling thread until deployied
   * compenents are started. If timeout for starting expires then Exception is thrown.
   *
   * @param earFile           the file which is transfered on the local machine and points to the ear.
   *                          It has to represent relative file path from test root directory.
   * @param getRemoteSupports if true then gets a jar file containing all classes that client application
   *                          would need to run the specified application.
   * @return                  name of the deployed component.
   *
   * @exception   Exception   If the component could not be deployed
   *                          or errors occurred while reading archive, for example
   *                          inproper xml format of ear descriptor or component descriptors.
   */
  public String deployEar(String earFile, boolean getRemoteSupports) throws Exception;


  /**
   * Deploys component found in the archive file using specified remote support( placed in default test properties
   * or properties which test has passed to DeployEnvironemnt) and applying given properties necessary for deployment.
   * This method blocks calling thread until deployied compenents are started.
   * If timeout for starting expires then Exception is thrown.
   *
   * @param earFile           the file which is transfered on the local machine and points to the ear.
   *                          It has to represent relative file path from test root directory.
   * @param p                 properties holding information necessary for the process of deployment;
   *                          they indicate J2EE Specification type, root lookup, container type.
   * @param getRemoteSupports if true then gets a jar file containing all classes that client application
   *                          would need to run the specified application.
   * @return                  name of the deployed component.
   *
   * @exception   Exception   If some of the components could not be deployed
   *                          or errors occurred while reading archive, for example
   *                          inproper xml format of ear descriptor or component descriptors.
   */
  public String deployEar(String earFile, boolean getRemoteSupports, Properties p) throws Exception;

  /**
   * Updates previously deployed component found in the ear file.
   * This method blocks calling thread until deployied compenents are started.
   * If timeout for starting expires then Exception is thrown.
   *
   * @param earFile           the name of the ear that contains the update information.
   *                          It has to represent relative file path from test root directory.
   * @param getRemoteSupports if true then gets a jar file containing all classes that client application
   *                          would need to run the specified application.
   * @return                  name of updated component
   * @throws Exception        If component could not be updated
   *                          or errors occured while reading archive, for example
   *                          inproper xml format of ear descriptor or component descriptors.
   */
  public String update(String earFile, boolean getRemoteSupports) throws Exception;

  /**
   * Updates previously deployed component found in the ear file applying given properties necessary for update.
   * This method blocks calling thread until deployied compenents are started.
   * If timeout for starting expires then Exception is thrown.
   *
   * @param earFile           the name of the ear that contains the update information.
   *                          It has to represent relative file path from test root directory.
   * @param getRemoteSupports if true then gets a jar file containing all classes that client application
   *                          would need to run the specified application.
   * @param props             properties holding information necessary for the process of update;
   *                          they indicate J2EE Specification type, root lookup, container type, etc.
   * @return                  name of updated component
   * @throws Exception        If component could not be updated
   *                          or errors occured while reading archive, for example
   *                          inproper xml format of ear descriptor or component descriptors.
   */
  public String update(String earFile, boolean getRemoteSupports, Properties props) throws Exception;

  /**
   * Updates previously deployed components found in the ear file applying given properties necessary for update.
   * This method blocks calling thread until deployied compenents are started.
   * If timeout for starting expires then Exception is thrown.
   *
   * @param earFile           the name of the ear that contains the update information.
   *                          It has to represent relative file path from test root directory.
   * @param props             properties holding information necessary for the process of update;
   *                          they indicate J2EE Specification type, root lookup, container type, etc.
   * @param getRemoteSupports if true then gets a jar file containing all classes that client application
   *                          would need to run the specified application.
   *
   * @return                  array of names of the updated components concatanated with
   *                          dash and String representation of their type.
   *
   * @throws   Exception      If some of the components could not be updated
   *                          or errors occured while reading archive, for example
   *                          inproper xml format of ear descriptor or component descriptors.
   */
  public String[] updateComponents(String earFile, Properties props , boolean getRemoteSupports) throws Exception;


  /**
   * Updates components found in the archive file to the specified container using given remote support
   * and applying properties necessary for deployment.
   *
   * @param archiveFile    the name of the archive that is being deployed.
   * @param containerName  the name of the container for which this archive is. If it is null, deploy service searches for a suitable container using its mechanism for automatically recognition of the suitable container.
   * @param remoteSupport  the remote support that this component will have.
   * @param props          properties holding information necessary for the process of deployment;
   *                       they indicate J2EE Specification type, root lookup, container type, application name of the dummy application which is created for this stand alone module
   *
   * @return               array of names of the deployed components concatenated with
   *                       dash and String representation of their type.
   *
   * @exception   Exception 
   *
   */
  public String[] update(String archiveFile, String containerName, String[] remoteSupport, Properties props) throws Exception;

  /**
   * Removes the specified application from servers in cluster.
   *
   * @param application the name of the application that will be removed.
   *
   * @throws Exception  if the some of the components of the application could not be removed.
   */
  public void remove(String application) throws Exception;

  /**
   * Deploys a library. The jars contains all the needed information about library.
   * @deprecated   the name of the library is now part of the provider.xml inside the jar
   *               use method deployLibrary(String libName , String jar).
   *
   * @param libName    optional library name, by default is the name of the jar.
   * @param jars       deployment jars of a library.
   * @throws Exception thrown if a problem during deploy process occurs.
   */
  public void deployLibrary(String libName, String[] jars) throws Exception;

  /**
   * Deploys a library. The jar contains all the needed information about library.
   *
   * @param      libName   optional library name, by default is the name of the jar.
   * @param      jar       deployment jar of a library.
   *                       It has to represent relative file path from test root directory.
   * @throws  Exception thrown if a problem during deploy process occurs.
   */
  public void deployLibrary(String libName , String jar) throws Exception;

  /**
   * Deprecated for 6.30.Method throws Exception if is used.
   * @param libName
   * @param jars
   * @throws Exception
   */
  public void updateLibrary(String libName, String[] jars) throws Exception;

  /**
   * Removes an existing library with the specified name from the server.
   *
   * @param libName     name of this library.
   *
   * @throws  Exception thrown if a problem during remove process occurs.
   */
  public void removeLibrary(String libName) throws Exception;
  /**
   * Makes references from this application to the specified components.
   * Each reference means that an application needs all the
   * referenced components for its regular work.
   *
   * @param   fromApplication name of application.
   * @param   toLibraries     Properties with key - the component name, and value - type of the reference : hard or weak. These are the valid component names:
   *                          for application - the application name
   *                          for service - service:<service_name>
   *                          for interface - interface:<interface_name>
   *                          for library - library:<library_name>
   *
   * @throws  Exception       thrown if a problem during making references process occurs.
   */
  public void makeReferences(String fromApplication, String[] toLibraries) throws Exception;
  /**
   * Removes references from this application to the specified libraries.
   *
   * @param   fromApplication      name of application.
   * @param   toLibraries          list of library names references to which will be removed.
   *
   * @exception  Exception   thrown if a problem during removing references process occurs.
   */
  public void removeReferences(String fromApplication, String[] toLibraries) throws Exception;

  /**
   * This method returns the status of the application, which is deployed on the server, that process the request.
   * Application status can be:
   * STOPPED, STARTED, STARTING, STOPPING, UPGRADING, UNKNOWN.
   *
   * @param applicationName   the name of the application, which status will be returned.
   * @return                  String representation of application status.
   * @throws Exception        if a problem during getting application status occurs.
   */

  public String getApplicationStatus(String applicationName) throws Exception;
  /**
   * Stops an application so that it will not be accessible for any actions from client side.
   * Only existing application can be stopped. This method does not remove the application from server though
   * application is not accessible until it is started again.
   *
   * @param applicationName name of application to be stopped.
   * @throws Exception      thrown if a remote problem during stop process occurs.
   */
  public void stopApplication(String applicationName) throws Exception;

  /**
   * Stops an application on the specified servers so that it will not be accessible for any actions from client side.
   * Only existing application can be stopped. This method does not remove the application from server though
   * application is not accessible until it is started again.
   * This method has asynchronous implementation, i.e.
   * its execution does not guarantee the completion of stopping process.
   *
   * @param       applicationName         name of application to be stopped.
   *
   * @param       serverNames             names of the servers on which the application will be stopped. If this parameter is null, the application will be stopped in the whole cluster.
   *
   * @exception   Exception         thrown if a remote problem during stop process occurs.
   */
  public void stopApplication(String applicationName, String[] serverNames) throws Exception;

  /**
   * Starts an application so that to be accessible for the clients.
   * Only existing applications that are in status "STOPPED" can be started.
   *
   * @param applicationName name of application to be started.
   * @throws Exception      thrown if a remote problem during start process occurs
   *                        or the specified application is currently not deployed
   *                        or if timeout of starting the application expires.
   */
  public void startApplication(String applicationName) throws Exception;

  /**
   * Starts an application on the specified servers so that to be accessible for the clients.
   * Only existing applications that are in status "STOPPED" or "IMPLICIT_STOPPED" can be started.
   * This method has asynchronous implementation, i.e.
   * its execution does not guarantee the completion of starting process.
   *
   * @param       applicationName         name of application to be started.
   *
   * @param       serverNames             names of the servers on which the application will be started. If this parameter is null, the application will be started in the whole cluster.
   *
   * @exception   Exception         thrown if a remote problem during start process occurs
   *                                      or the specified application is currently not deployed.
   */
  public void startApplication(String applicationName, String[] serverNames) throws Exception;



    /**
   * Updates previously deployed application with the given single files that are part of it.
   *
   * @param files     contains information about the changed files for the application.
   * @param appName   the name of the application which is being updated.
   * @param props     properties holding information necessary for the process of update.
   *                  It is container dependent.
   *
   * @exception   RemoteException
   *                               If some of the components could not be updated
   *                               or errors occurred while reading archive, for example
   *                               improper xml format of ear descriptor or component descriptors.
   */
  public void singleFileUpdate(FileUpdateInfo[] files, String appName, Properties props) throws Exception;


}
