package com.inqmy.ats;

import java.sql.*;
import java.io.File;

import com.sap.engine.services.dbpool.deploy.DataSourceManager;

/**
 * This interface represents an environment for working with database.
 * Database properties such as IP, user name, user password and so on
 * are taken from Environment property. Here simple example :
 * <pre>
 *  import java.util.Properties;
 *  import com.inqmy.ats.Test;
 *  import com.inqmy.ats.DatabaseEnvironment;
 *  import com.inamy.ats.system.EnvironmentFactory;
 *
 *  public class MyTest implements com.inqmy.ats.Test {
 *
 *    public int test() {
 *      try {
 *        int result = 0;
 *        Properties props = new Properties();
 *        props.setProperty("db_user","myUser");
 *        props.setProperty("db_password","myPassword");
 *        DatabaseEnvironment db = (DatabaseEnvironment)EnvironmentFactory.getEnvironment(props , EnvironmentFactory.DATABASE);
 *        String st = "CREATE TABLE COFFEES "+
 *                    "(COF_NAME VARCHAR(32), SUP_ID INTEGER, PRICE FLOAT, "+
 *                    "SALES INTEGER, TOTAL INTEGER)";
 *        db.executeUpdate(st);
 *        ...................
 *        String query = "SELECT COF_NAME, PRICE FROM COFFEES";
 *        ResultSet resultSet = db.executeQuery(query);
 *        ...................
 *
 *        //here goes some kind of test logic
 *        ...................
 *        return result;
 *      } catch(Exception exc) {
 *        exc.printStackTrace();
 *        return 0;
 *      }
 *    }
 *
 *  }
 * </pre>
 */
public interface DatabaseEnvironment extends Environment {

  /**
   * Deploys JDBC driver with the specified name and driver files. It is deployed as a library. Each
   * library is a name-value pair. The name is a String identifier of the library and
   * the value contains an array of paths to all files belonging to this library.
   *
   * @param driverName     name of this driver.
   *
   * @param driverFiles    list of serializable files. Files  are specified by strings
   * representing the paths to the files.
   *
   * @exception  Exception  thrown if a problem during register process occurs.
   */
  public void deployJdbcDriver(String driverName, File[] driverFiles) throws Exception;

  public DataSourceManager getDataSourceManager();

  /**
   * Removes an existing driver with the specified name from the server/servers in the cluster.
   *
   * @param  driverName     name of this driver.
   *
   * @exception  Exception  thrown if a problem during remove process occurs.
   */
  public void undeployJdbcDriver(String driverName) throws Exception;


  public String getDefaultDSName() throws Exception;


  /**
   * Creates a data source depending on the specified name , maxConnection and defConnection
   * count.Method uses database settings in passed Properties.
   *
   * @param name  name of the datasource to create.
   * @param maxConection max count of physical connection that can be made to Database.
   * @param defConection default count of physical connections to Databes.
   *
   * @return  The Application name for this datasource
   *
   * @exception  Exception thrown if a problem during process occurs.
   */

  public String createDataSource(String name, int maxConection, int defConection) throws Exception;

  /**
   * Creates a data source depending on the specified name with max connections count 2 and
   * default onnections count 1.Method uses database settings in passed Properties.
   *
   * @param name  name of the datasource to create.
   *
   * @return  The Application name for this datasource
   *
   * @exception  Exception thrown if a problem during process occurs.
   */

  public String createDataSource(String name) throws Exception;

  public String createAliasForDataSource(String dataSourceName , String[] aliases) throws Exception;
  /**
   *
   * @param appName
   * @throws Exception
   */
  public void removeApplication(String appName) throws Exception;

  /**
   * Makes connection to database throught driver, creates Statement
   * object and calls executeUpdate(String sql) on it (INSERT, UPDATE or DELETE statement).
   *
   * @param sql         sql statements to execute.Statements in this parameter are separated by ';' symbol.
   *                    If statement starts with symbol '#' it is skipped.
   * @throws Exception  if there is problem executing quiery.
   */
  public void executeUpdate(String sql) throws Exception;
  /**
   * Makes connection to database throught driver, creates Statement
   * object and calls executeQuiery(String sql) on it.
   *
   * @param            query typically this is a static SQL SELECT statement
   * @return           a ResultSet object that contains the data produced by the given query; never null.
   * @throws Exception if error occurs.
   */
  public ResultSet executeQuery(String query) throws Exception;

}
