/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.tomcat.internal;


import java.io.*;
import java.lang.reflect.*;
import java.util.*;

import org.apache.jasper.compiler.*;
import org.eclipse.core.runtime.*;

public class JdtJasperCompiler implements JavaCompiler {
	private static final String JDT_COMPILER_CLASS =
		"org.eclipse.jdt.internal.compiler.batch.Main";
	private static final String JDT_CORE_PLUGIN = "org.eclipse.jdt.core";
	private static Class jdtCompiler = null;
	private static Method compileMethod = null;
	private String outputDir;
	boolean classDebugInfo;
	private String classpath;
	private String encoding;
	private OutputStream os;
	private PrintWriter writer;
	private StringBuffer arguments;

	public JdtJasperCompiler() {
		arguments = new StringBuffer();
	}

	/*
	 * @see JavaCompiler#compile(String)
	 */
	public boolean compile(String java) {

		String src = arg(java);

		if (src == null || src.trim().equals("")) {
			log("JDT Compiler: source file not specified");
			return false;
		}

		// append standard flags
		if (outputDir != null && !outputDir.trim().equals("")) {
			arguments.append(" -d ");
			arguments.append(arg(outputDir));
		}

		if (classpath != null && !classpath.trim().equals("")) {
			arguments.append(" -classpath ");
			arguments.append(classpath);
		}
		
		if (classDebugInfo) {
			arguments.append(" -g");
		}		

		arguments.append(" ");
		arguments.append(src);

		try {
			//Main.compile(args, writer);
			return doCompile();
		} catch (Exception e) {
			log("JDT Compiler: " + e.toString());
			return false;
		}
	}

	/*
	 * @see JavaCompiler#setClassDebugInfo(boolean)
	 */
	public void setClassDebugInfo(boolean classDebugInfo) {
		this.classDebugInfo = classDebugInfo;
	}
	
	/*
	 * @see JavaCompiler#setClasspath(String)
	 */
	public void setClasspath(String classpath) {
		// Need to enclose each path with blanks by ""
		// NOTE: BUG in JDT compiler. We need to enclose all the members by ""
		// The bug does not allow mixing strings enclosed by " with non-enclosed ones.
		this.classpath = "";
		StringTokenizer st = new StringTokenizer(classpath, File.pathSeparator);
		while (st.hasMoreTokens()) {
			String s = st.nextToken();
			this.classpath += "\"" + s + "\"" + File.pathSeparator;
		}

		// add the java runtime
		String rtJar =
			System.getProperty("java.home")
				+ File.separator
				+ "lib"
				+ File.separator
				+ "rt.jar";
		this.classpath += "\"" + rtJar + "\"";
	}

	/*
	 * @see JavaCompiler#setCompilerPath(String)
	 */
	public void setCompilerPath(String compilerpath) {
		// do nothing ... using built-in jdt compiler
	}

	/*
	 * @see JavaCompiler#setEncoding(String)
	 */
	public void setEncoding(String encoding) {
		this.encoding = arg(encoding);
	}

	/*
	 * @see JavaCompiler#setMsgOutput(OutputStream)
	 */
	public void setMsgOutput(OutputStream os) {
		this.os = os;
		if (os != null)
			this.writer = new PrintWriter(os);
	}

	/*
	 * @see JavaCompiler#setOutputDir(String)
	 */
	public void setOutputDir(String outputDir) {
		this.outputDir = arg(outputDir);
	}

	private void log(String msg) {
		if (writer != null) {
			writer.println(msg);
			writer.flush();
		}
	}

	private String arg(String arg) {
		if (arg == null)
			return arg;

		String s = arg.trim();
		if (s.indexOf(" ") != -1)
			s = "\"" + s + "\"";
		return s;
	}

	/**
	 * Returns the compile method from the jdt batch compiler.
	 * Use static variables to cache some of the info for faster execution
	 */
	private Method getCompileMethod() {
		// cache the compiler
		if (compileMethod != null)
			return compileMethod;

		if (jdtCompiler == null) {
			IPluginRegistry registry = Platform.getPluginRegistry();
			IPluginDescriptor jdtCorePlugin = registry.getPluginDescriptor(JDT_CORE_PLUGIN);
			if (jdtCorePlugin == null)
				return null;

			try {
				jdtCompiler =
					jdtCorePlugin.getPluginClassLoader().loadClass(JDT_COMPILER_CLASS);
				if (jdtCompiler == null)
					throw new Exception(JDT_COMPILER_CLASS + " not found");
			} catch (Exception e) {
				return null;
			}
		}

		// at this point the jdtCompiler is not null
		try {
			compileMethod =
				jdtCompiler.getMethod(
					"compile",
					new Class[] { String.class, PrintWriter.class, PrintWriter.class });
			return compileMethod;
		} catch (Exception e) {
			return null;
		}

	}

	private boolean doCompile() {
		try {
			final String args = arguments.toString();
			compileMethod = getCompileMethod();
			if (compileMethod == null)
				throw new Exception(JDT_COMPILER_CLASS + ".compile() not found ");

			compileMethod.invoke(jdtCompiler, new Object[] { args, writer, writer });
			return true;
		} catch (Exception e) {
			log("JDT Compiler: " + e.toString());
			return false;
		}
	}

}
