/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.transaction;

import com.sap.netweaver.bc.rf.util.enum.*;

/**
 * Enumeration of transaction states.
 *
 * @author Vedran Lerenc
 * @created 23. Januar 2003
 */
public final class Status extends EnumWithDescription {
  /*
   * Unknown transaction state.
   */
  public final static Status UNKNOWN =
    new Status(1, "Unknown");

  /*
   * Transaction not yet begun.
   */
  public final static Status NO_TRANSACTION = new Status(6, "No Transaction");

  /*
   * Transaction ongoing.
   */
  public final static Status ACTIVE =
    new Status(2, "Active");

  /*
   * Transaction marked for rollback, i.e. can't be committed anymore. Commit
   * will fail on a transaction marked for rollback.
   */
  public final static Status MARKED_ROLLBACK =
    new Status(5, "Marked Rollback");

  /*
   * First phase of two phase commit (2PC) protocol, where the coordinator
   * asks all participants for their preparation vote. Preparation means
   * storing all after-images of the transaction into the redo buffer in
   * stable memory. When all participants return successful in time, the whole
   * transaction will be committed, otherwise it will be rolled back. There is
   * a time of uncertainity when the participant has voted but doesn't get the
   * coordinators decision. When the participant fails in this time, it has to
   * recover, i.e. to await a repair before continuing, because the
   * coordinator may either have choosen to commit or roll back the
   * transaction. During that time the participant is blocked until the
   * repair is done. The coordinator is never uncertain, but may also fail
   * while issuing its decision to all of the participants. During that time
   * all participants not gotton the decision are blocked until the
   * coordinator recovers and sends/resends the decision to them.
   */
  public final static Status REQUESTING_COMMIT =
    new Status(9, "Requesting Commit");

  /*
   * First phase of two phase commit (2PC) protocol, where the participant
   * prepares, but not yet commits the transaction. Preparation means storing
   * all after-images of the transaction into the redo buffer in stable
   * memory. After this state it will move into the prepared state, which is
   * still the first phase of the 2PC.
   */
  public final static Status PREPARING_COMMIT =
    new Status(9, "Preparing Commit");

  /*
   * First phase of two phase commit (2PC) protocol, where the participant is
   * preared, but has not yet commited the transaction. Preparation means
   * storing all after-images of the transaction into the redo buffer in
   * stable memory. After this state it awaits either a commit or rollback
   * message.
   */
  public final static Status PREPARED_COMMIT =
    new Status(10, "Prepared Commit");

  /*
   * Second phase of two phase commit (2PC) protocol, where the
   * transaction gets done, i.e. committed, i.e. gets written to stable
   * storage. After this state it will move into the committed state, which
   * too is the second phase of the 2PC.
   */
  public final static Status COMMITTING =
    new Status(4, "Committing");

  /*
   * Second phase of two phase commit (2PC) protocol, where the
   * transaction is done, i.e. committed, i.e. is written to stable storage.
   * After this state the transaction won't change it's state anymore.
   */
  public final static Status COMMITED =
    new Status(3, "Commited");

  /*
   * Second phase of two phase commit (2PC) protocol, where the
   * transaction gets aborted, i.e. rolled back, i.e. gets removed from stable
   * storage. After this state it will move into the rolled back state, which
   * too is the second phase of the 2PC.
   */
  public final static Status ROLLING_BACK =
    new Status(8, "Rolling Back");

  /*
   * Second phase of two phase commit (2PC) protocol, where the
   * transaction is aborted, i.e. rolled back, i.e. is removed from stable
   * storage. After this state the transaction won't change it's state
   * anymore.
   */
  public final static Status ROLLED_BACK =
    new Status(7, "Rolled Back");

  /**
   * Construct object of class Status.
   *
   * @param id enumeration id
   * @param description enumeration description
   */
  private Status(final int id, final String description) {
    super(id, description, false);
  }
}
