/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.transaction;

import com.sap.netweaver.bc.rf.util.enum.*;

/**
 * Enumeration of isolation levels.
 *
 * @author Vedran Lerenc
 * @created 23. Januar 2003
 */
public final class IsolationLevel extends EnumWithDescription {
  /*
   * Unknown isolation level.
   */
  public final static IsolationLevel UNKNOWN = new IsolationLevel(0, "Unknown");

  /*
   * This isolation level just ensures that no physically corrupt data will
   * read. It won't issue any shared locks while reading data. The consequence
   * is, that you can read uncommitted data from ongoing transactions. This
   * data might get committed or rolled back later. This is the lowest and
   * computationally cheapest isolation level also called "dirty read".
   */
  public final static IsolationLevel READ_UNCOMMITTED = new IsolationLevel(1, "Read Uncommitted");

  /*
   * This isolation level ensures that no physically corrupt data will read and
   * that no data will be read that is not yet committed. It will
   * issue shared locks while reading data. The consequence is, that you won't
   * read uncommitted data, but that you can read data that might get changed
   * before the end of the transaction by other committed transactions. This
   * isolation level prevents you from "dirty reads", but not from "non-
   * repeatable reads" and "phantom reads" (see below).
   */
  public final static IsolationLevel READ_COMMITTED = new IsolationLevel(2, "Read Committed");

  /*
   * This isolation level ensures that no physically corrupt data will read and
   * that no data will be read that is not yet committed and that once read
   * data won't change. It will issue exclusive locks while reading data that
   * is used in a query. The consequence is, that no other transaction can
   * update the data - the data is blocked for other transactions. This
   * isolation level prevents you from both, the "dirty reads" and the "non-
   * repeatable reads", but not from "phantom reads" (see below).
   */
  public final static IsolationLevel REPEATABLE_READ = new IsolationLevel(3, "Repeatable Read");

  /*
   * This isolation level ensures that no physically corrupt data will read and
   * that no data will be read that is not yet committed and that once read
   * data won't change and that no data will be additionally seen when
   * re-executing queries on the system. It will issue exclusive locks on the
   * data and the whole data set used in a query. The consequence is, that no
   * other transaction can update the data or data set - the data set with
   * it's content is fully blocked for other transactions. This isolation
   * level prevents you from all of the above, the "dirty reads", the "non-
   * repeatable reads" and the "phantom reads". "Phantom reads", e.g. occur
   * when a transaction re-executes a query returning a set of rows that
   * differs from a set of rows returned in a previous query, due to another
   * recently-committed transaction.
   */
  public final static IsolationLevel SERIALIZABLE = new IsolationLevel(4, "Serializable");

  /**
   * Construct object of class TransactionStatus.
   *
   * @param id enumeration id
   * @param description enumeration description
   */
  private IsolationLevel(final int id, final String description) {
    super(id, description, false);
  }
}
