/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.security.acl;

import java.util.*;

/**
 * An <code>IAclPermission</code> denominates a set of actions that can be
 * carried out on an object and can be granted or denied to a principal. <br>
 *
 * <ul>
 *   <li> the mapping (IAction, IAclPermission) is done in the <code>
 *   IAclSecurityManager</code> </li>
 *   <li> <code>IAclPermission<code>s can be nested, so an <code>IAclPermission
 *   <code> may contain other <code>IAclPermission<code>s as members</li>
 * </ul>
 *
 *
 * @created 20. Mrz 2003
 */
public interface IAclPermission {
  /**
   * permissions: full control always contains all other permissions
   */
  public final static String ACL_PERMISSION_FULL_CONTROL = "fullcontrol";

  /**
   * repository permissions
   */
  public final static String ACL_PERMISSION_READ = "read";
  public final static String ACL_PERMISSION_WRITE = "write";
  public final static String ACL_PERMISSION_READWRITE = "readwrite";
  public final static String ACL_PERMISSION_DELETE = "delete";

  /**
   * service permissions
   */
  public final static String ACL_PERMISSION_SUBSCRIPTION_UNSUBSCRIBE = "sub_unsubscribe";
  public final static String ACL_PERMISSION_SUBSCRIPTION_ACTIVE = "sub_active";
  public final static String ACL_PERMISSION_SUBSCRIPTION_PASSIVE = "sub_passive";
  public final static String ACL_PERMISSION_SUBSCRIPTION_FOLDER = "sub_folder";
  public final static String ACL_PERMISSION_CHANGE_APPROVER = "changeappr";
  public final static String ACL_PERMISSION_DELETE_COLLABORATION = "deletecollab";
  public final static String ACL_PERMISSION_FOLDERSETTINGS = "foldersettings";

  /**
   * @return The atomic value
   * @exception AclPersistenceException Exception raised in failure situation
   * @supplierCardinality 1
   * @directed
   */
  /*
   * #IAclPermissionList lnkIAclPermissionList;
   */
  /**
   * check whether the present permission has no members
   *
   * @return The atomic value
   * @return The atomic value
   * @return The atomic value
   * @return The atomic value
   * @return true iff the permission denominates
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public boolean isAtomic()
    throws AclPersistenceException;// returns true iff the permission contains only a single action

  /**
   * check whether some permission is a member of the present permission
   *
   * @param permission a permission
   * @return true iff the permission is a member of the present permission
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isMember(IAclPermission permission)
    throws AclPersistenceException;

  /**
   * get a list of the members of the present permission
   *
   * @return the members of the present permission
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IAclPermissionList getMembers()
    throws AclPersistenceException;

  /**
   * add some permission as a new member to the present permission
   *
   * @param permission the permission
   * @return true iff the permission was added successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception AclPermissionExistsException Exception raised in failure
   *      situation
   * @exception AclPredefinedPermissionException Exception raised in failure
   *      situation
   * @exception AclPermissionUsedException Exception raised in failure situation
   */
  public boolean addMember(IAclPermission permission)
    throws AclPersistenceException, AclPermissionExistsException, AclPredefinedPermissionException, AclPermissionUsedException;

  /**
   * remove a member from the present permission
   *
   * @param permission the member permission
   * @return true iff the member permission was removed successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception AclPredefinedPermissionException Exception raised in failure
   *      situation
   * @exception AclPermissionUsedException Exception raised in failure situation
   */
  public boolean removeMember(IAclPermission permission)
    throws AclPersistenceException, AclPredefinedPermissionException, AclPermissionUsedException;

  /**
   * check whether the present permission is predefined
   *
   * @return true iff the present permission is predefined
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isPredefined()
    throws AclPersistenceException;

  /**
   * check whether a permission and all of its members are contained in the
   * present permission
   *
   * @param permission permission to be checked
   * @return true iff a permission and all of its members are contained in the
   *      present permission
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isContained(IAclPermission permission)
    throws AclPersistenceException;

  /**
   * get the name of the permission
   *
   * @return the name of the permission
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public String getName()
    throws AclPersistenceException;

  /**
   * get the description of the permission
   *
   * @return the description of the permission
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public String getDescription()
    throws AclPersistenceException;

  /**
   * Get the description attribute of the IAclPermission object.
   *
   * @param locale locale of description
   * @return The description value
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public String getDescription(Locale locale)
    throws AclPersistenceException;

  /**
   * get the sort index of the permission (for display in a control)
   *
   * @return the sort index of the permission
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public int getSortIndex()
    throws AclPersistenceException;

  /**
   * check whether some permission equals the present permission
   *
   * @param permission the permission to compare to
   * @return true iff the permission equals the present permission
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public boolean equals(IAclPermission permission)
    throws AclPersistenceException;
}
