/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.security.acl;

import com.sap.security.api.*;

/**
 * Interface describing an access control list entry, short ACL entry. Each
 * resource may have an ACL with many ACL entries. An entry describes for one
 * principal (may be a user, group or whatever is considered an entity) a
 * permission this principal has on the given resource. The ACL itself is just
 * an envelope for many ACL entries. </p> An ACL entry may define a permission
 * this principal has on the given resource, but it may also define a permission
 * a principal is denied to have (so called negative ACL entry). </p> ACLs are
 * propagated down from above the resource hierarchy on the fly, i.e. they are
 * not really set on the descendants but derived when accessed. All descendants
 * not having an own ACL inherit one from their next legator having one.
 * Legators are ancestors responsible for the security permissions of their
 * descendants. </p>
 *
 * @author Vedran Lerenc
 * @created 31. Januar 2003
 */
public interface IAclEntry {
  /**
   * Get principal of the ACL entry.
   *
   * @return principal of the ACL entry
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public IPrincipal getPrincipal()
    throws AclPersistenceException;

  /**
   * Check whether or not the given permission is covered by this ACL entry.
   *
   * @param permission permission to be checked
   * @return true if the given permission is covered by this ACL entry
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public boolean checkPermission(IAclPermission permission)
    throws AclPersistenceException;

  /**
   * Get permission of the ACL entry.
   *
   * @return permission of the ACL entry
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public IAclPermission getPermission()
    throws AclPersistenceException;

  /**
   * get the sort index of the ACL entry - the sequence of the ACEs in the ACL
   * is controlled by the sort indices - the sequence is only relevant if the
   * IResourceAclManager supports negative ACEs (check with the
   * areNegativeAclEntriesSupported() method)
   *
   * @return the sort index
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public int getSortIndex()
    throws AclPersistenceException;

  /**
   * set the sort index of the ACL entry
   *
   * @param sortIndex sort index
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception AclNotAuthorizedException Exception raised in failure situation
   */
  public void setSortIndex(int sortIndex)
    throws AclPersistenceException, AclNotAuthorizedException;

  /**
   * Check whether or not this ACL entry denies its permission.
   *
   * @return true if this ACL entry denies its permission
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public boolean isNegative()
    throws AclPersistenceException;

  /**
   * Check wether or not the ACL entry is read only.
   *
   * @return true if the ACL entry is read only
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public boolean isReadOnly()
    throws AclPersistenceException;
}
