/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.security.acl;

import com.sap.netweaver.bc.rf.common.*;
import com.sap.netweaver.bc.rf.common.exception.*;

import com.sap.security.api.*;

import java.util.*;

/**
 * Interface describing an access control list, short ACL. Each resource may
 * have an ACL with many ACL entries. An entry describes for one principal (may
 * be a user, group or whatever is considered an entity) a permission this
 * principal has on the given resource. The ACL itself is just an envelope for
 * many ACL entries. </p> An ACL entry may define a permission this principal
 * has on the given resource, but it may also define a permission a principal is
 * denied to have (so called negative ACL entry). </p> ACLs are propagated down
 * from above the resource hierarchy on the fly, i.e. they are not really set on
 * the descendants but derived when accessed. All descendants not having an own
 * ACL inherit one from their next legator having one. Legators are ancestors
 * responsible for the security permissions of their descendants. </p>
 *
 * @author Vedran Lerenc
 * @created 31. Januar 2003
 */
public interface IAcl {
  /**
   * Get resource handle this ACL is assigned to.
   *
   * @return resource handle this ACL is assigned to
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public IResourceHandle getResource()
    throws AclPersistenceException;

  /**
   * Get list of the owners of this ACL.
   *
   * @return list of the owners of this ACL
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public List getOwners()
    throws AclPersistenceException;

  /**
   * Add new owner to this ACL.
   *
   * @param principal principal to be added as owner
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception ResourceException when the resource handle was invalid
   * @exception AclNotAuthorizedException Exception raised in failure situation
   */
  public void addOwner(IPrincipal principal)
    throws AclPersistenceException, AclNotAuthorizedException, ResourceException;

  /**
   * Remove owner from the this ACL.
   *
   * @param principal principal to be removed as owner
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception ResourceException when the resource handle was invalid
   * @exception AclNotAuthorizedException Exception raised in failure situation
   * @exception AclLastOwnerException Exception raised in failure situation
   */
  public void removeOwner(IPrincipal principal)
    throws AclPersistenceException, AclNotAuthorizedException, AclLastOwnerException, ResourceException;

  /**
   * Check whether or not the given principal is an owner of this ACL.
   *
   * @param principal principal to be checked for being owner
   * @return true when the given principal is an owner of this ACL
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public boolean isOwner(IPrincipal principal)
    throws AclPersistenceException;

  /**
   * Check whether or not the given principal is authorized to modify this ACL.
   * Beside owners also super principals not being an owner are authorized to
   * modify an ACL.
   *
   * @param principal principal to be checked for being authorized
   * @return true if the given principal is authorized to modify this ACL
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public boolean isAuthorized(IPrincipal principal)
    throws AclPersistenceException;

  /**
   * Get list of ACL entries.
   *
   * @return list of ACL entries
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public List getEntries()
    throws AclPersistenceException;

  /**
   * Get list of ACL entries for a given principal.
   *
   * @param principal principal
   * @return list of ACL entries for a given principal
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public List getEntries(IPrincipal principal)
    throws AclPersistenceException;

  /**
   * Add ACL entry to this ACL.
   *
   * @param aclEntry ACL entry
   * @return true if the ACL entry was added successfully
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception ResourceException when the resource handle associated with the
   *      ACL was invalid
   * @exception AclNotAuthorizedException Exception raised in failure situation
   * @exception AclAlreadyAssignedToAclException Exception raised in failure
   *      situation
   * @exception AclPermissionNotSupportedException Exception raised in failure
   *      situation
   */
  public boolean addEntry(IAclEntry aclEntry)
    throws AclPersistenceException, AclNotAuthorizedException, AclAlreadyAssignedToAclException, AclPermissionNotSupportedException, ResourceException;

  /**
   * Remove ACL entry from this ACL.
   *
   * @param aclEntry ACL entry
   * @return true if the ACL entry was removed successfully
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception ResourceException when the resource handle associated with the
   *      ACL was invalid
   * @exception AclNotAuthorizedException Exception raised in failure situation
   */
  public boolean removeEntry(IAclEntry aclEntry)
    throws AclPersistenceException, AclNotAuthorizedException, ResourceException;

  /**
   * Check whether or not the ACL grants a specific permission to a principal.
   *
   * @param principal principal to be checked
   * @param permission permission to be checked
   * @return true if the permission is granted
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public boolean checkPermission(IPrincipal principal, IAclPermission permission)
    throws AclPersistenceException;

  /**
   * Get principal which has locked the ACL.
   *
   * @return principal which has locked the ACL
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public IPrincipal getLockingUser()
    throws AclPersistenceException;

  /**
   * Lock the ACL.
   *
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception AclNotAuthorizedException Exception raised in failure situation
   */
  public void lock()
    throws AclPersistenceException, AclNotAuthorizedException;

  /**
   * Unlock the ACL.
   *
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception AclNotAuthorizedException Exception raised in failure situation
   */
  public void unlock()
    throws AclPersistenceException, AclNotAuthorizedException;

  /**
   * Check whether or not the ACL is locked.
   *
   * @return true if the ACL is locked
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public boolean isLocked()
    throws AclPersistenceException;

  /**
   * Check wether or not the ACL is read only.
   *
   * @return true if the ACL is read only
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public boolean isReadOnly()
    throws AclPersistenceException;
}
