/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.security;

import com.sap.netweaver.bc.rf.util.flyweight.*;
import com.sap.netweaver.bc.rf.util.resource.*;

import com.sap.tc.logging.*;

import java.io.*;
import java.util.*;

/**
 * Class implementing permission.
 *
 * @created 23. Januar 2003
 */
public class Permission
   extends FlyWeight
   implements IPermission, Cloneable, Serializable {

  // Permission name
  protected final String name, localName;

  // Resource bundle
  private ResourceBundles resourceBundles;

  // Leaf permissions
  public final static IPermission PERMISSION_READ_CONTENT =
    new Permission("leaf_read_content");
  public final static IPermission PERMISSION_READ_PROPERTIES =
    new Permission("leaf_read_properties");
  public final static IPermission PERMISSION_WRITE_CONTENT =
    new Permission("leaf_write_content");
  public final static IPermission PERMISSION_WRITE_PROPERTIES =
    new Permission("leaf_write_properties");
  public final static IPermission PERMISSION_DELETE =
    new Permission("leaf_delete");

  // Node permissions
  public final static IPermission PERMISSION_LIST =
    new Permission("node_list_children");
  public final static IPermission PERMISSION_CREATE =
    new Permission("node_create_child");
  public final static IPermission PERMISSION_READ_NODE_PROPERTIES =
    new Permission("node_read_properties");
  public final static IPermission PERMISSION_WRITE_NODE_PROPERTIES =
    new Permission("node_write_properties");
  public final static IPermission PERMISSION_DELETE_NODE =
    new Permission("node_delete");
  private static Location logger = Location.getLocation(com.sap.netweaver.bc.rf.common.security.Permission.class);

  // Permission name namespace
  private final static String PERMISSION_NAMESPACE =
    "http://sapportals.com/xmlns/cm";

  // Static resource bundle for predefined permissions
  // TBD: Missing resource bundle base (name) for predefined permissions
  private static ResourceBundles PREDEFINED_RESOURCEBUNDLE =
    ResourceBundles.getBundle("");


  /**
   * Construct object of class Permission.
   *
   * @param resourceBundles resource bundle
   * @param fullName
   */
  protected Permission(
    final String fullName,
    ResourceBundles resourceBundles) {
    // Call flyweight constructor
    super(fullName, false);

    // Store name and resource bundle
    this.localName = null;
    this.name = (String)getFlyWeightId();
    this.resourceBundles = resourceBundles;
  }


  /**
   * Construct object of class Permission.
   *
   * @param localName
   */
  private Permission(final String localName) {
    // Call flyweight constructor
    super(PERMISSION_NAMESPACE + localName, false);

    // Store name and resource bundle
    this.localName = localName;
    this.name = (String)getFlyWeightId();
    this.resourceBundles = PREDEFINED_RESOURCEBUNDLE;
  }


  /**
   * Get unique name of permission. IPermission implementations are handled by
   * one security sub manager for which they must be implemented and be unique.
   *
   * @return unique name of permission
   */
  public String getName() {
    return name;
  }


  /**
   * Get description of permission.
   *
   * @return description of permission
   */
  public String getDescription() {
    String description = null;
    try {
      if ((resourceBundles != null) && (resourceBundles.exists(name))) {
        description = resourceBundles.getString(name);
      }
    }
    catch (MissingResourceException exception) {
      // Do nothing; standard behaviour will return permission name
      logger.warningT(
        "getDescription(137)",
        "Requested description on permission named "
         + name
         + " but found no valid description despite a given resource bundle!");
    }
    if (description == null) {
      return name;
    }
    return description;
  }


  /**
   * Get localized description of permission.
   *
   * @param locale locale to be used for localization
   * @return localized description of permission
   */
  public String getDescription(Locale locale) {
    String description = null;
    try {
      if ((resourceBundles != null) && (resourceBundles.exists(name))) {
        description = resourceBundles.getString(name, locale);
      }
    }
    catch (MissingResourceException exception) {
      // Do nothing; standard behaviour will return permission name
      logger.warningT(
        "getDescription(165)",
        "Requested localized description ("
         + locale.getDisplayName()
         + ") on permission named "
         + name
         + " but found no valid description despite a given resource bundle!");
    }
    if (description == null) {
      return name;
    }
    return description;
  }


  /**
   * Get descriptive text for this permission.
   *
   * @return descriptive text for this permission
   */
  public String toString() {
    return name;
  }


  /**
   * Get hashcode for this permission.
   *
   * @return hashcode for this permission
   */
  public final int hashCode() {
    // Since the super() call to the FlyWeight class is slower,
    // just implement the base class equals() method directly
    return name.hashCode();
  }


  /**
   * Compare this and another object for equality. Works only fine when handling
   * objects of classes loaded by same class classloader.
   *
   * @param obj object to be compared against
   * @return result of comparison
   */
  public final boolean equals(final Object obj) {
    // Since the super() call to the Object class is slower,
    // just implement the default equals() method directly
    return this == obj;
  }


  /**
   * Clone this permission. The clone method will always return the same
   * instance for faster equality comparison. This is not harmful, since the
   * permission is immutable.
   *
   * @return this permission
   */
  public final Object clone() {
    // Return this instance - no clones are allowed due to identify check
    return this;
  }


  /**
   * Get permission (all permissions are cached for performance and
   * internationalization reasons and indexed by their full name). If it isn't
   * available, it won't be created, because it's internationalization
   * information must be provided by the permission originator first before
   * clients are allowed to us it.
   *
   * @param fullName permission full name
   * @return
   */
  public static IPermission getPermission(final String fullName) {
    // Retrieve flyweight permission based on class and id
    return (IPermission)getFlyWeightInherited(Permission.class, fullName);
  }

  public String getLocalName() {
    return this.localName;
  }
}
