/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.security;

import com.sap.netweaver.bc.rf.common.*;
import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.util.exception.*;

/**
 * Raised if the authorization failed that means the user in the context does
 * not have the required permissions. In addition this excpetion can be raised
 * if an attempt is made to modify a locked or read-only resource.
 *
 * @created 14. April 2003
 */
public class AccessDeniedException extends ResourceException {
  // Name of the permission that was not granted
  private String permissionName = null;

  // Id of the user which had not the permission
  private String userId = null;

  // Definition of default auto exception log behaviour,
  // used when no user behaviour is supplied during exception construction
  private final static boolean LOG_EXCEPTION_ON_CREATION = true;


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   */
  public AccessDeniedException(IRid rid, String permissionName, String userId) {
    super(rid, LOG_EXCEPTION_ON_CREATION);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public AccessDeniedException(IRid rid, String permissionName, String userId, boolean logExceptionOnCreation) {
    super(rid, logExceptionOnCreation);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param rootCause throwable object which caused this exception
   */
  public AccessDeniedException(IRid rid, String permissionName, String userId, Throwable rootCause) {
    super(rid, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public AccessDeniedException(IRid rid, String permissionName, String userId, Throwable rootCause, boolean logExceptionOnCreation) {
    super(rid, rootCause, logExceptionOnCreation);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param message not localized message
   */
  public AccessDeniedException(IRid rid, String permissionName, String userId, String message) {
    super(rid, message, LOG_EXCEPTION_ON_CREATION);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   */
  public AccessDeniedException(IRid rid, String permissionName, String userId, String message, Object[] args) {
    super(rid, message, args, LOG_EXCEPTION_ON_CREATION);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   */
  public AccessDeniedException(IRid rid, String permissionName, String userId, String message, Throwable rootCause) {
    super(rid, message, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   */
  public AccessDeniedException(IRid rid, String permissionName, String userId, String message, Object[] args, Throwable rootCause) {
    super(rid, message, args, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param message not localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public AccessDeniedException(IRid rid, String permissionName, String userId, String message, boolean logExceptionOnCreation) {
    super(rid, message, logExceptionOnCreation);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public AccessDeniedException(IRid rid, String permissionName, String userId,
    String message,
    Object[] args,
    boolean logExceptionOnCreation) {
    super(rid, message, args, logExceptionOnCreation);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public AccessDeniedException(IRid rid, String permissionName, String userId,
    String message,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, message, rootCause, logExceptionOnCreation);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public AccessDeniedException(IRid rid, String permissionName, String userId,
    String message,
    Object[] args,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, message, args, rootCause, logExceptionOnCreation);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   */
  protected AccessDeniedException(IRid rid, String permissionName, String userId, ResourceAccessor resourceAccessor, String key) {
    super(rid, resourceAccessor, key, LOG_EXCEPTION_ON_CREATION);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   */
  protected AccessDeniedException(IRid rid, String permissionName, String userId,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args) {
    super(rid, resourceAccessor, key, args, LOG_EXCEPTION_ON_CREATION);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param rootCause throwable object which caused this exception
   */
  protected AccessDeniedException(IRid rid, String permissionName, String userId,
    ResourceAccessor resourceAccessor,
    String key,
    Throwable rootCause) {
    super(rid, resourceAccessor, key, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   */
  protected AccessDeniedException(IRid rid, String permissionName, String userId,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    Throwable rootCause) {
    super(rid, resourceAccessor, key, args, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected AccessDeniedException(IRid rid, String permissionName, String userId,
    ResourceAccessor resourceAccessor,
    String key,
    boolean logExceptionOnCreation) {
    super(rid, resourceAccessor, key, logExceptionOnCreation);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected AccessDeniedException(IRid rid, String permissionName, String userId,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    boolean logExceptionOnCreation) {
    super(rid, resourceAccessor, key, args, logExceptionOnCreation);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected AccessDeniedException(IRid rid, String permissionName, String userId,
    ResourceAccessor resourceAccessor,
    String key,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, resourceAccessor, key, rootCause, logExceptionOnCreation);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected AccessDeniedException(IRid rid, String permissionName, String userId,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, resourceAccessor, key, args, rootCause, logExceptionOnCreation);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param message not localized message
   */
  protected AccessDeniedException(IRid rid, String permissionName, String userId, ResourceMessage message) {
    super(rid, message, LOG_EXCEPTION_ON_CREATION);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   */
  protected AccessDeniedException(IRid rid, String permissionName, String userId, ResourceMessage message, Throwable rootCause) {
    super(rid, message, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param message not localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected AccessDeniedException(IRid rid, String permissionName, String userId, ResourceMessage message, boolean logExceptionOnCreation) {
    super(rid, message, logExceptionOnCreation);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Construct object of class AccessDeniedException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param permissionName name of the permission that was not granted
   * @param userId id of the user which had not the permission
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected AccessDeniedException(IRid rid, String permissionName, String userId,
    ResourceMessage message,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, message, rootCause, logExceptionOnCreation);
    this.permissionName = permissionName;
    this.userId = userId;
  }


  /**
   * Get name of the permission that was not granted.
   *
   * @return name of the permission that was not granted
   */
  public String getPermissionName() {
    return this.permissionName;
  }


  /**
   * Get id of the user which had not the permission.
   *
   * @return id of the user which had not the permission
   */
  public String getUserId() {
    return this.userId;
  }
}
