/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.property.xmltype;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.common.property.*;

import java.util.*;

/**
 * Class implementing a Xml single value property. See {@link IProperty} and
 * {@link IMutableProperty} for details on properties
 *
 * @author Vedran Lerenc
 * @created 29. Januar 2003
 * @see AbstractProperty
 * @see IMutableProperty
 * @see IProperty
 */
public class MutableXmlSingleValueProperty
   extends XmlSingleValueProperty
   implements IMutableProperty {
  /**
   * Construct property instance/copy based on another property template
   * (uaually a property to be changed in parts).
   *
   * @param property property used as template
   * @exception ResourceException when the property creation failed
   */
  public MutableXmlSingleValueProperty(XmlSingleValueProperty property)
    throws ResourceException {
    super(property.getPropertyName(), property.getValue(), property.getAttributes(), property.getFlags(), null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param value property value
   * @exception ResourceException when the property creation failed
   */
  public MutableXmlSingleValueProperty(IPropertyName name, String value)
    throws ResourceException {
    super(name, value, null, 0, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param value property value
   * @param attributes property attributes
   * @exception ResourceException when the property creation failed
   */
  public MutableXmlSingleValueProperty(IPropertyName name, String value, Properties attributes)
    throws ResourceException {
    super(name, value, attributes, 0, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param value property value
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @exception ResourceException when the property creation failed
   */
  public MutableXmlSingleValueProperty(IPropertyName name, String value, int flags)
    throws ResourceException {
    super(name, value, null, flags, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param value property value
   * @param attributes property attributes
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @exception ResourceException when the property creation failed
   */
  public MutableXmlSingleValueProperty(IPropertyName name, String value, Properties attributes, int flags)
    throws ResourceException {
    super(name, value, attributes, flags, null);
  }

  /**
   * Set property value.
   *
   * @param value property value
   */
  public void setValue(String value) {
    this.value = value;
  }

  /**
   * Set all attributes for the property at once. The given attribute map will
   * be stored directly (as is - without being copied). Note that this call
   * disconnects the attribute map returned from a getAttributes() call from the
   * property.
   *
   * @param attributes property map holding all attributes
   * @exception ResourceException when the attributes map or the property is
   *      invalid
   */
  public void setAttributes(Properties attributes)
    throws ResourceException {
    this.attributes = attributes;
  }

  /**
   * Remove all attributes for the property at once.
   *
   * @exception ResourceException when the property is invalid
   */
  public void clearAttributes()
    throws ResourceException {
    if (attributes != null) {
      attributes.clear();
    }
  }

  /**
   * Get attribute for the property.
   *
   * @param attributeName attribute name
   * @return attribute value
   * @exception ResourceException when the attribute name or the property is
   *      invalid
   */
  public String getAttribute(String attributeName)
    throws ResourceException {
    if (attributes != null) {
      return attributes.getProperty(attributeName);
    }
    return null;
  }

  /**
   * Add attribute for the property.
   *
   * @param attributeName attribute name
   * @param attributeValue attribute value
   * @exception ResourceException when the attribute name or value or the
   *      property is invalid
   */
  public void setAttribute(String attributeName, String attributeValue)
    throws ResourceException {
    if (attributes == null) {
      attributes = new Properties();
    }
    attributes.setProperty(attributeName, attributeValue);
  }

  /**
   * Remove attribute from the property.
   *
   * @param attributeName attribute name
   * @exception ResourceException when the attribute name or the property is
   *      invalid
   */
  public void removeAttribute(String attributeName)
    throws ResourceException {
    if (attributes != null) {
      attributes.remove(attributeName);
    }
  }
}
