/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.property.inttype;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.common.property.*;
import com.sap.netweaver.bc.rf.util.resource.*;

import java.util.*;

/**
 * Class implementing a int single value property. See {@link IProperty} and
 * {@link IMutableProperty} for details on properties
 *
 * @author Vedran Lerenc
 * @created 29. Januar 2003
 * @see AbstractProperty
 * @see IMutableProperty
 * @see IProperty
 */
public class IntSingleValueProperty
   extends AbstractProperty {
  protected int value;

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param value property value
   * @exception ResourceException when the property creation failed
   */
  public IntSingleValueProperty(IPropertyName name, int value)
    throws ResourceException {
    this(name, value, null, 0, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param value property value
   * @param attributes property attributes
   * @exception ResourceException when the property creation failed
   */
  public IntSingleValueProperty(IPropertyName name, int value, Properties attributes)
    throws ResourceException {
    this(name, value, attributes, 0, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param value property value
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @exception ResourceException when the property creation failed
   */
  public IntSingleValueProperty(IPropertyName name, int value, int flags)
    throws ResourceException {
    this(name, value, null, flags, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param value property value
   * @param attributes property attributes
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @exception ResourceException when the property creation failed
   */
  public IntSingleValueProperty(IPropertyName name, int value, Properties attributes, int flags)
    throws ResourceException {
    this(name, value, attributes, flags, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param value property value
   * @param attributes property attributes
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @param resourceBundles resource bundles containing (localized) property
   *      description
   * @exception ResourceException when the property creation failed
   */
  protected IntSingleValueProperty(
    IPropertyName name,
    int value,
    Properties attributes,
    int flags,
    ResourceBundles resourceBundles)
    throws ResourceException {
    super(name, attributes, flags, resourceBundles);
    this.value = value;
  }

  /**
   * Get immutable type of this property.
   *
   * @return immutable type of this property
   */
  public PropertyType getType() {
    return PropertyType.INT;
  }

  /**
   * Returns true if this property is a multi-value property.
   *
   * @return true if this property is a multi-value property
   */
  public boolean isMultiValue() {
    return false;
  }

  /**
   * Get the property value converted to a string. If this is a multi value
   * property just return the first value.
   *
   * @return property value converted to a string
   * @exception ResourceException when the property is invalid
   */
  public String getValueAsString()
    throws ResourceException {
    return String.valueOf(value);
  }

  /**
   * Get the property values converted to a string array. If this is a single
   * value property just return an array with the value as only element.
   *
   * @return array of property values converted to strings
   * @exception ResourceException when the property is invalid
   */
  public String[] getValuesAsStrings()
    throws ResourceException {
    String[] result = new String[1];
    result[0] = String.valueOf(value);
    return result;
  }

  /**
   * Get property value.
   *
   * @return property value
   */
  public int getValue() {
    return value;
  }
}
