/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.property.datetype;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.common.property.*;

import java.util.*;

/**
 * Class implementing a Date multi value property. See {@link IProperty} and
 * {@link IMutableProperty} for details on properties.
 *
 * @author Vedran Lerenc
 * @created 29. Januar 2003
 * @see AbstractProperty
 * @see IMutableProperty
 * @see IProperty
 */
public class MutableDateMultiValueProperty
   extends DateMultiValueProperty
   implements IMutableProperty {
  /**
   * Construct property instance/copy based on another property template
   * (uaually a property to be changed in parts).
   *
   * @param property property used as template
   * @exception ResourceException when the property creation failed
   */
  public MutableDateMultiValueProperty(DateMultiValueProperty property)
    throws ResourceException {
    super(property.getType(), property.getPropertyName(), property.getValues(), property.getAttributes(), property.getFlags(), null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param propertyType specific property type
   * @param name property name
   * @param values property values
   * @exception ResourceException when the property creation failed
   */
  public MutableDateMultiValueProperty(PropertyType propertyType, IPropertyName name, Date[] values)
    throws ResourceException {
    super(propertyType, name, values, null, 0, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param propertyType specific property type
   * @param name property name
   * @param values property values
   * @param attributes property attributes
   * @exception ResourceException when the property creation failed
   */
  public MutableDateMultiValueProperty(
    PropertyType propertyType,
    IPropertyName name,
    Date[] values,
    Properties attributes)
    throws ResourceException {
    super(propertyType, name, values, attributes, 0, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param propertyType specific property type
   * @param name property name
   * @param values property values
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @exception ResourceException when the property creation failed
   */
  public MutableDateMultiValueProperty(PropertyType propertyType, IPropertyName name, Date[] values, int flags)
    throws ResourceException {
    super(propertyType, name, values, null, flags, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param propertyType specific property type
   * @param name property name
   * @param values property values
   * @param attributes property attributes
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @exception ResourceException when the property creation failed
   */
  public MutableDateMultiValueProperty(
    PropertyType propertyType,
    IPropertyName name,
    Date[] values,
    Properties attributes,
    int flags)
    throws ResourceException {
    super(propertyType, name, values, attributes, flags, null);
  }

  /**
   * Set array of property values. The given values array will be stored
   * directly (as is - without being copied). Note that this call disconnects
   * the values array returned from a getValues() call from the property.
   *
   * @param values array of property values
   */
  public void setValues(Date[] values) {
    this.values = values;
  }

  /**
   * Set array of property values. The given values array will be stored
   * directly (as is - without being copied). Note that this call disconnects
   * the values array returned from a getValues() call from the property.
   *
   * @param values array of property values
   */
  public void setValuesAsLongs(long[] values) {
    this.values = new Date[values.length];
    for (int i = 0; i < this.values.length; i++) {
      this.values[i] = new Date(values[i]);
    }
  }

  /**
   * Set property value with given index.
   *
   * @param value property value
   * @param index index of property value
   * @exception IndexOutOfBoundsException when the index is out of bounds
   */
  public void setValue(Date value, int index)
    throws IndexOutOfBoundsException {
    values[index] = value;
  }

  /**
   * Set property value with given index.
   *
   * @param value property value
   * @param index index of property value
   * @exception IndexOutOfBoundsException when the index is out of bounds
   */
  public void setValueAsLong(long value, int index)
    throws IndexOutOfBoundsException {
    setValue(new Date(value), index);
  }

  /**
   * Add property value. The array will be reallocated and copied, so beware of
   * the performance drawback. Note that this call disconnects the values array
   * returned from a getValues() call from the property.
   *
   * @param value property value
   */
  public void addValue(Date value) {
    Date[] newValues = new Date[values.length + 1];
    for (int i = 0; i < values.length; i++) {
      newValues[i] = values[i];
    }
    newValues[values.length] = value;
    values = newValues;
  }

  /**
   * Add property value. The array will be reallocated and copied, so beware of
   * the performance drawback. Note that this call disconnects the values array
   * returned from a getValues() call from the property.
   *
   * @param value property value
   */
  public void addValueAsLong(long value) {
    addValue(new Date(value));
  }

  /**
   * Insert property value at given index. The array will be reallocated and
   * copied, so beware of the performance drawback. Note that this call
   * disconnects the values array returned from a getValues() call from the
   * property.
   *
   * @param value property value
   * @param index index of property value
   * @exception IndexOutOfBoundsException when the index is out of bounds
   */
  public void insertValue(Date value, int index)
    throws IndexOutOfBoundsException {
    if ((index >= 0) && (index < values.length)) {
      Date[] newValues = new Date[values.length + 1];
      for (int i = 0; i < index; i++) {
        newValues[i] = values[i];
      }
      newValues[index] = value;
      for (int i = index; i < values.length; i++) {
        newValues[i + 1] = values[i];
      }
      values = newValues;
    }
    else {
      throw new IndexOutOfBoundsException(
        "Index "
         + index
         + " is out of bounds of Date multi value property with "
         + values.length
         + " elements!");
    }
  }

  /**
   * Insert property value at given index. The array will be reallocated and
   * copied, so beware of the performance drawback. Note that this call
   * disconnects the values array returned from a getValues() call from the
   * property.
   *
   * @param value property value
   * @param index index of property value
   * @exception IndexOutOfBoundsException when the index is out of bounds
   */
  public void insertValueAsLong(long value, int index)
    throws IndexOutOfBoundsException {
    insertValue(new Date(value), index);
  }

  /**
   * Remove all occurences of the given property value. The array will be
   * reallocated and copied, so beware of the performance drawback. Note that
   * this call disconnects the values array returned from a getValues() call
   * from the property.
   *
   * @param value property value
   */
  public void removeValue(Date value) {
    int[] indexes = new int[values.length];
    int count = 0;
    for (int i = 0; i < values.length; i++) {
      if (values[i] == value) {
        indexes[count++] = i;
      }
    }
    if (count > 0) {
      Date[] newValues = new Date[values.length - count];
      int indexesIndex = 0;
      int actualIndex = 0;
      for (; indexesIndex < count; indexesIndex++) {
        for (; actualIndex < indexes[indexesIndex]; actualIndex++) {
          newValues[actualIndex - indexesIndex] = values[actualIndex];
        }
        actualIndex++;
      }
      for (; actualIndex < values.length; actualIndex++) {
        newValues[actualIndex - indexesIndex] = values[actualIndex];
      }
      values = newValues;
    }
  }

  /**
   * Remove all occurences of the given property value. The array will be
   * reallocated and copied, so beware of the performance drawback. Note that
   * this call disconnects the values array returned from a getValues() call
   * from the property.
   *
   * @param value property value
   */
  public void removeValueAsLong(long value) {
    removeValue(new Date(value));
  }

  /**
   * Remove property value with given index. The array will be reallocated and
   * copied, so beware of the performance drawback. Note that this call
   * disconnects the values array returned from a getValues() call from the
   * property.
   *
   * @param index index of property value
   * @exception IndexOutOfBoundsException when the index is out of bounds
   */
  public void removeValue(int index)
    throws IndexOutOfBoundsException {
    if ((index >= 0) && (index < values.length)) {
      Date[] newValues = new Date[values.length - 1];
      for (int i = 0; i < index; i++) {
        newValues[i] = values[i];
      }
      for (int i = index; i < newValues.length; i++) {
        newValues[i] = values[i + 1];
      }
      values = newValues;
    }
    else {
      throw new IndexOutOfBoundsException(
        "Index "
         + index
         + " is out of bounds of Date multi value property with "
         + values.length
         + " elements!");
    }
  }

  /**
   * Set all attributes for the property at once. The given attribute map will
   * be stored directly (as is - without being copied). Note that this call
   * disconnects the attribute map returned from a getAttributes() call from the
   * property.
   *
   * @param attributes property map holding all attributes
   * @exception ResourceException when the attributes map or the property is
   *      invalid
   */
  public void setAttributes(Properties attributes)
    throws ResourceException {
    this.attributes = attributes;
  }

  /**
   * Remove all attributes for the property at once.
   *
   * @exception ResourceException when the property is invalid
   */
  public void clearAttributes()
    throws ResourceException {
    if (attributes != null) {
      attributes.clear();
    }
  }

  /**
   * Get attribute for the property.
   *
   * @param attributeName attribute name
   * @return attribute value
   * @exception ResourceException when the attribute name or the property is
   *      invalid
   */
  public String getAttribute(String attributeName)
    throws ResourceException {
    if (attributes != null) {
      return attributes.getProperty(attributeName);
    }
    return null;
  }

  /**
   * Add attribute for the property.
   *
   * @param attributeName attribute name
   * @param attributeValue attribute value
   * @exception ResourceException when the attribute name or value or the
   *      property is invalid
   */
  public void setAttribute(String attributeName, String attributeValue)
    throws ResourceException {
    if (attributes == null) {
      attributes = new Properties();
    }
    attributes.setProperty(attributeName, attributeValue);
  }

  /**
   * Remove attribute from the property.
   *
   * @param attributeName attribute name
   * @exception ResourceException when the attribute name or the property is
   *      invalid
   */
  public void removeAttribute(String attributeName)
    throws ResourceException {
    if (attributes != null) {
      attributes.remove(attributeName);
    }
  }
}
