/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.property.booleantype;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.common.property.*;

import java.util.*;

/**
 * Class implementing a boolean multi value property. See {@link IProperty} and
 * {@link IMutableProperty} for details on properties.
 *
 * @author Vedran Lerenc
 * @created 29. Januar 2003
 * @see AbstractProperty
 * @see IMutableProperty
 * @see IProperty
 */
public class MutableBooleanMultiValueProperty
   extends BooleanMultiValueProperty
   implements IMutableProperty {
  /**
   * Construct property instance/copy based on another property template
   * (uaually a property to be changed in parts).
   *
   * @param property property used as template
   * @exception ResourceException when the property creation failed
   */
  public MutableBooleanMultiValueProperty(BooleanMultiValueProperty property)
    throws ResourceException {
    super(property.getPropertyName(), property.getValues(), property.getAttributes(), property.getFlags(), null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param values property values
   * @exception ResourceException when the property creation failed
   */
  public MutableBooleanMultiValueProperty(IPropertyName name, boolean[] values)
    throws ResourceException {
    super(name, values, null, 0, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param values property values
   * @param attributes property attributes
   * @exception ResourceException when the property creation failed
   */
  public MutableBooleanMultiValueProperty(
    IPropertyName name,
    boolean[] values,
    Properties attributes)
    throws ResourceException {
    super(name, values, attributes, 0, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param values property values
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @exception ResourceException when the property creation failed
   */
  public MutableBooleanMultiValueProperty(IPropertyName name, boolean[] values, int flags)
    throws ResourceException {
    super(name, values, null, flags, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param values property values
   * @param attributes property attributes
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @exception ResourceException when the property creation failed
   */
  public MutableBooleanMultiValueProperty(
    IPropertyName name,
    boolean[] values,
    Properties attributes,
    int flags)
    throws ResourceException {
    super(name, values, attributes, flags, null);
  }

  /**
   * Set array of property values. The given values array will be stored
   * directly (as is - without being copied). Note that this call disconnects
   * the values array returned from a getValues() call from the property.
   *
   * @param values array of property values
   */
  public void setValues(boolean[] values) {
    this.values = values;
  }

  /**
   * Set property value with given index.
   *
   * @param value property value
   * @param index index of property value
   * @exception IndexOutOfBoundsException when the index is out of bounds
   */
  public void setValue(boolean value, int index)
    throws IndexOutOfBoundsException {
    values[index] = value;
  }

  /**
   * Add property value. The array will be reallocated and copied, so beware of
   * the performance drawback. Note that this call disconnects the values array
   * returned from a getValues() call from the property.
   *
   * @param value property value
   */
  public void addValue(boolean value) {
    boolean[] newValues = new boolean[values.length + 1];
    for (int i = 0; i < values.length; i++) {
      newValues[i] = values[i];
    }
    newValues[values.length] = value;
    values = newValues;
  }

  /**
   * Insert property value at given index. The array will be reallocated and
   * copied, so beware of the performance drawback. Note that this call
   * disconnects the values array returned from a getValues() call from the
   * property.
   *
   * @param value property value
   * @param index index of property value
   * @exception IndexOutOfBoundsException when the index is out of bounds
   */
  public void insertValue(boolean value, int index)
    throws IndexOutOfBoundsException {
    if ((index >= 0) && (index < values.length)) {
      boolean[] newValues = new boolean[values.length + 1];
      for (int i = 0; i < index; i++) {
        newValues[i] = values[i];
      }
      newValues[index] = value;
      for (int i = index; i < values.length; i++) {
        newValues[i + 1] = values[i];
      }
      values = newValues;
    }
    else {
      throw new IndexOutOfBoundsException(
        "Index "
         + index
         + " is out of bounds of boolean multi value property with "
         + values.length
         + " elements!");
    }
  }

  /**
   * Remove property value with given index. The array will be reallocated and
   * copied, so beware of the performance drawback. Note that this call
   * disconnects the values array returned from a getValues() call from the
   * property.
   *
   * @param index index of property value
   * @exception IndexOutOfBoundsException when the index is out of bounds
   */
  public void removeValue(int index)
    throws IndexOutOfBoundsException {
    if ((index >= 0) && (index < values.length)) {
      boolean[] newValues = new boolean[values.length - 1];
      for (int i = 0; i < index; i++) {
        newValues[i] = values[i];
      }
      for (int i = index; i < newValues.length; i++) {
        newValues[i] = values[i + 1];
      }
      values = newValues;
    }
    else {
      throw new IndexOutOfBoundsException(
        "Index "
         + index
         + " is out of bounds of boolean multi value property with "
         + values.length
         + " elements!");
    }
  }

  /**
   * Set all attributes for the property at once. The given attribute map will
   * be stored directly (as is - without being copied). Note that this call
   * disconnects the attribute map returned from a getAttributes() call from the
   * property.
   *
   * @param attributes property map holding all attributes
   * @exception ResourceException when the attributes map or the property is
   *      invalid
   */
  public void setAttributes(Properties attributes)
    throws ResourceException {
    this.attributes = attributes;
  }

  /**
   * Remove all attributes for the property at once.
   *
   * @exception ResourceException when the property is invalid
   */
  public void clearAttributes()
    throws ResourceException {
    if (attributes != null) {
      attributes.clear();
    }
  }

  /**
   * Add attribute for the property.
   *
   * @param attributeName attribute name
   * @param attributeValue attribute value
   * @exception ResourceException when the attribute name or value or the
   *      property is invalid
   */
  public void setAttribute(String attributeName, String attributeValue)
    throws ResourceException {
    if (attributes == null) {
      attributes = new Properties();
    }
    attributes.setProperty(attributeName, attributeValue);
  }

  /**
   * Remove attribute from the property.
   *
   * @param attributeName attribute name
   * @exception ResourceException when the attribute name or the property is
   *      invalid
   */
  public void removeAttribute(String attributeName)
    throws ResourceException {
    if (attributes != null) {
      attributes.remove(attributeName);
    }
  }
}
