/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.property.booleantype;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.common.property.*;
import com.sap.netweaver.bc.rf.util.resource.*;

import java.util.*;

/**
 * Class implementing a boolean multi value property. See {@link IProperty} and
 * {@link IMutableProperty} for details on properties.
 *
 * @author Vedran Lerenc
 * @created 29. Januar 2003
 * @see AbstractProperty
 * @see IMutableProperty
 * @see IProperty
 */
public class BooleanMultiValueProperty
   extends AbstractProperty {
  protected boolean[] values;

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param values property values
   * @exception ResourceException when the property creation failed
   */
  public BooleanMultiValueProperty(IPropertyName name, boolean[] values)
    throws ResourceException {
    this(name, values, null, 0, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param values property values
   * @param attributes property attributes
   * @exception ResourceException when the property creation failed
   */
  public BooleanMultiValueProperty(
    IPropertyName name,
    boolean[] values,
    Properties attributes)
    throws ResourceException {
    this(name, values, attributes, 0, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param values property values
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @exception ResourceException when the property creation failed
   */
  public BooleanMultiValueProperty(IPropertyName name, boolean[] values, int flags)
    throws ResourceException {
    this(name, values, null, flags, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param values property values
   * @param attributes property attributes
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @exception ResourceException when the property creation failed
   */
  public BooleanMultiValueProperty(
    IPropertyName name,
    boolean[] values,
    Properties attributes,
    int flags)
    throws ResourceException {
    this(name, values, attributes, flags, null);
  }

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param values property values
   * @param attributes property attributes
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @param resourceBundles resource bundles containing (localized) property
   *      description
   * @exception ResourceException when the property creation failed
   */
  protected BooleanMultiValueProperty(
    IPropertyName name,
    boolean[] values,
    Properties attributes,
    int flags,
    ResourceBundles resourceBundles)
    throws ResourceException {
    super(name, attributes, flags, resourceBundles);
    this.values = values;
  }

  /**
   * Get immutable type of this property.
   *
   * @return immutable type of this property
   */
  public PropertyType getType() {
    return PropertyType.BOOLEAN;
  }

  /**
   * Returns true if this property is a multi-value property.
   *
   * @return true if this property is a multi-value property
   */
  public boolean isMultiValue() {
    return true;
  }

  /**
   * Get the property value converted to a string. If this is a multi value
   * property just return the first value.
   *
   * @return property value converted to a string
   * @exception ResourceException when the property is invalid
   */
  public String getValueAsString()
    throws ResourceException {
    if (values.length > 0) {
      return String.valueOf(values[0]);
    }
    else {
      return null;
    }
  }

  /**
   * Get the property values converted to a string array. If this is a single
   * value property just return an array with the value as only element.
   *
   * @return array of property values converted to strings
   * @exception ResourceException when the property is invalid
   */
  public String[] getValuesAsStrings()
    throws ResourceException {
    String[] result = new String[values.length];
    for (int i = 0; i < result.length; i++) {
      result[i] = String.valueOf(values[i]);
    }
    return result;
  }

  /**
   * Get array of property values. The returned values array will be the stored
   * one (as is - without being copied). Note that the stored values array is no
   * longer connected to the property when it is set by a setValues() call or
   * completely exchanged otherwise.
   *
   * @return array of property values
   */
  public boolean[] getValues() {
    return values;
  }

  /**
   * Get property value with given index.
   *
   * @param index index of property value
   * @return property value with given index
   * @exception IndexOutOfBoundsException when the index is out of bounds
   */
  public boolean getValue(int index)
    throws IndexOutOfBoundsException {
    return values[index];
  }
}
