/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.property;

import com.sap.netweaver.bc.rf.common.*;
import com.sap.netweaver.bc.rf.util.exception.*;

/**
 * Class implementing exception raised when a property value is out of bounds.
 *
 * @created 31. Januar 2003
 */
public class PropertyNameTooLongException extends AbstractPropertyException {
  // Namespace length constraint.
  private int namespaceMaxLength = 0;

  // Local name length constraint.
  private int localNameMaxLength = 0;

  // Definition of default auto exception log behaviour,
  // used when no user behaviour is supplied during exception construction
  private final static boolean LOG_EXCEPTION_ON_CREATION = true;

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   */
  public PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength) {
    super(rid, propertyName, LOG_EXCEPTION_ON_CREATION);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength, boolean logExceptionOnCreation) {
    super(rid, propertyName, logExceptionOnCreation);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param rootCause throwable object which caused this exception
   */
  public PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength, Throwable rootCause) {
    super(rid, propertyName, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength, Throwable rootCause, boolean logExceptionOnCreation) {
    super(rid, propertyName, rootCause, logExceptionOnCreation);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param message not localized message
   */
  public PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength, String message) {
    super(rid, propertyName, message, LOG_EXCEPTION_ON_CREATION);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   */
  public PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength, String message, Object[] args) {
    super(rid, propertyName, message, args, LOG_EXCEPTION_ON_CREATION);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   */
  public PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength, String message, Throwable rootCause) {
    super(rid, propertyName, message, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   */
  public PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength, String message, Object[] args, Throwable rootCause) {
    super(rid, propertyName, message, args, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param message not localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength, String message, boolean logExceptionOnCreation) {
    super(rid, propertyName, message, logExceptionOnCreation);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength,
    String message,
    Object[] args,
    boolean logExceptionOnCreation) {
    super(rid, propertyName, message, args, logExceptionOnCreation);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength,
    String message,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, propertyName, message, rootCause, logExceptionOnCreation);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength,
    String message,
    Object[] args,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, propertyName, message, args, rootCause, logExceptionOnCreation);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   */
  protected PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength, ResourceAccessor resourceAccessor, String key) {
    super(rid, propertyName, resourceAccessor, key, LOG_EXCEPTION_ON_CREATION);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   */
  protected PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args) {
    super(rid, propertyName, resourceAccessor, key, args, LOG_EXCEPTION_ON_CREATION);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param rootCause throwable object which caused this exception
   */
  protected PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength,
    ResourceAccessor resourceAccessor,
    String key,
    Throwable rootCause) {
    super(rid, propertyName, resourceAccessor, key, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   */
  protected PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    Throwable rootCause) {
    super(rid, propertyName, resourceAccessor, key, args, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength,
    ResourceAccessor resourceAccessor,
    String key,
    boolean logExceptionOnCreation) {
    super(rid, propertyName, resourceAccessor, key, logExceptionOnCreation);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    boolean logExceptionOnCreation) {
    super(rid, propertyName, resourceAccessor, key, args, logExceptionOnCreation);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength,
    ResourceAccessor resourceAccessor,
    String key,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, propertyName, resourceAccessor, key, rootCause, logExceptionOnCreation);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, propertyName, resourceAccessor, key, args, rootCause, logExceptionOnCreation);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param message not localized message
   */
  protected PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength, ResourceMessage message) {
    super(rid, propertyName, message, LOG_EXCEPTION_ON_CREATION);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   */
  protected PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength, ResourceMessage message, Throwable rootCause) {
    super(rid, propertyName, message, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param message not localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength, ResourceMessage message, boolean logExceptionOnCreation) {
    super(rid, propertyName, message, logExceptionOnCreation);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Construct object of class PropertyNameTooLongException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param propertyName name of the property
   * @param namespaceMaxLength namespace length constraint
   * @param localNameMaxLength local name length constraint
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected PropertyNameTooLongException(IRid rid, IPropertyName propertyName, int namespaceMaxLength, int localNameMaxLength,
    ResourceMessage message,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, propertyName, message, rootCause, logExceptionOnCreation);
    this.localNameMaxLength = localNameMaxLength;
    this.namespaceMaxLength = namespaceMaxLength;
  }

  /**
   * Get namespace length constraint.
   *
   * @return namespace length constraint
   */
  public long getNamespaceMaxLength() {
    return this.namespaceMaxLength;
  }

  /**
   * Get local name length constraint.
   *
   * @return local name length constraint
   */
  public long getLocalNameMaxLength() {
    return this.localNameMaxLength;
  }
}
