/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.property;

import com.sap.netweaver.bc.rf.util.namespace.*;

import java.io.*;

/**
 * Class implementing a property name. This class predefines the system level
 * property names. Don't add any other then system property names here! Each
 * PropertyName follows the flyweight pattern for performance and memory
 * consumption reasons. When define own property names, you are part of that
 * pattern. When you define your own property name class, you should derive from
 * that class to profit from that pattern.
 *
 * @author Vedran Lerenc
 * @created 31. Januar 2003
 */
public class PropertyName extends Name implements IPropertyName, Cloneable, Serializable {
  // Property name namespace
  private final static String PROPERTYNAME_NAMESPACE = "http://sapportals.com/xmlns/cm";

  /**
   * Property describing the repository supplied resource name, i.e. the last
   * RID part
   */
  public final static IPropertyName RESOURCENAME = new PropertyName(PROPERTYNAME_NAMESPACE, "resourcename");

  /**
   * Property describing the repository supplied resource type
   */
  public final static IPropertyName RESOURCETYPE = new PropertyName(PROPERTYNAME_NAMESPACE, "resourcetype");

  /**
   * Property describing when the resource was created
   */
  public final static IPropertyName CREATED = new PropertyName(PROPERTYNAME_NAMESPACE, "created");

  /**
   * Property describing the user who created the resource
   */
  public final static IPropertyName CREATEDBY = new PropertyName(PROPERTYNAME_NAMESPACE, "createdby");

  /**
   * Property describing when the resource was last modified
   */
  public final static IPropertyName MODIFIED = new PropertyName(PROPERTYNAME_NAMESPACE, "modified");

  /**
   * Property describing who modified the resource last
   */
  public final static IPropertyName MODIFIEDBY = new PropertyName(PROPERTYNAME_NAMESPACE, "modifiedby");

  /**
   * Property describing whether or not the resource is a collection
   */
  public final static IPropertyName COLLECTION = new PropertyName(PROPERTYNAME_NAMESPACE, "collection");

  /**
   * Property describing of which link type a resource is (see namespace for
   * link type ids)
   */
  public final static IPropertyName LINKTYPE = new PropertyName(PROPERTYNAME_NAMESPACE, "linktype");

  /**
   * Property describing whether or not the resource is read only
   */
  public final static IPropertyName READONLY = new PropertyName(PROPERTYNAME_NAMESPACE, "readonly");

  /**
   * Property describing whether or not the resource is hidden
   */
  public final static IPropertyName HIDDEN = new PropertyName(PROPERTYNAME_NAMESPACE, "hidden");

  /**
   * Property describing the display name of the resource
   */
  public final static IPropertyName DISPLAYNAME = new PropertyName(PROPERTYNAME_NAMESPACE, "displayname");

  /**
   * Property describing the resource itself
   */
  public final static IPropertyName DESCRIPTION = new PropertyName(PROPERTYNAME_NAMESPACE, "description");

  /**
   * Property describing the content length of the resource
   */
  public final static IPropertyName CONTENTLENGTH = new PropertyName(PROPERTYNAME_NAMESPACE, "contentlength");

  /**
   * Property describing the content type of the resource
   */
  public final static IPropertyName CONTENTTYPE = new PropertyName(PROPERTYNAME_NAMESPACE, "contenttype");

  /**
   * Property describing the content encoding of the resource
   */
  public final static IPropertyName CONTENTENCODING = new PropertyName(PROPERTYNAME_NAMESPACE, "contentencoding");

  /**
   * Property describing the content language of the resource
   */
  public final static IPropertyName CONTENTLANGUAGE = new PropertyName(PROPERTYNAME_NAMESPACE, "contentlanguage");

  /**
   * Property describing an entity tag reflecting the <a
   * href="http://greenbytes.de/tech/webdav/rfc2616.html#rfc.section.3.11"> HTTP
   * entity tag</a> for the unfiltered content of this resource. If two strong
   * entity tags for a given resource are identical, the contents are identical
   * octet-by-octet. If two weak entitiy tags are identical, the contents are
   * semantically equivalent, but may differ in it's octet representation. Tags
   * are weak if and only if they start with a leading "W/". Note: The ETag of a
   * resource is not a unique identifier.
   */
  public final static IPropertyName ETAG = new PropertyName(PROPERTYNAME_NAMESPACE, "etag");

  /**
   * Construct property name based on namespace and local name (@link IName).
   *
   * @param namespace namespace
   * @param localName local name
   * @see IName
   */
  private PropertyName(final String namespace, final String localName) {
    // Call name constructor
    super(namespace, localName);
  }

  /**
   * Get property name (all property names are cached for performance reasons
   * and indexed by their full name). If it isn't available, it will be created
   * automatically.
   *
   * @param name property name
   * @return propertyName
   * @exception AbstractPropertyException Exception raised in failure situation
   */
  public static IPropertyName getPropertyName(final IName name)
    throws AbstractPropertyException {
    return getPropertyName(name.getFullName());
  }

  /**
   * Get property name (all property names are cached for performance reasons
   * and indexed by their full name). If it isn't available, it will be created
   * automatically.
   *
   * @param fullName property full name
   * @return propertyName
   * @exception AbstractPropertyException Exception raised in failure situation
   */
  public static IPropertyName getPropertyName(final String fullName)
    throws AbstractPropertyException {
    // Retrieve or create flyweight property name based on class and full name
    IPropertyName propertyName = (IPropertyName)getFlyWeight(PropertyName.class, fullName);
    if (propertyName == null) {
      // Check full name and thereby extract namespace and local name
      if (fullName == null) {
        throw new IllegalArgumentException("Name constraint failed: Full name must not be null!");
      }
      if (fullName.charAt(0) != '{') {
        throw new IllegalArgumentException(
          "Name constraint failed: Namespace in full name \""
           + fullName
           + "\" must be in curly brackets (begining bracket is missing)!");
      }
      int rightBracket = fullName.lastIndexOf('}');
      if (rightBracket < 0) {
        throw new IllegalArgumentException(
          "Name constraint failed: Namespace in full name \""
           + fullName
           + "\" must be in curly brackets (ending bracket is missing)!");
      }
      String namespace = fullName.substring(1, rightBracket);
      String localName = fullName.substring(rightBracket + 1);

      // Check contraints
      try {
        checkConstraints(namespace, localName);
      }
      catch (IllegalArgumentException exception) {
        //$JL-EXC$ 
        throw new PropertyNameInvalidException(null, null, exception.getMessage());
      }

      // Create property name
      try {
        propertyName = new PropertyName(namespace, localName);
      }
      catch (IllegalArgumentException exception) {
        if ((propertyName = (IPropertyName)getFlyWeight(PropertyName.class, fullName)) == null) {
          throw exception;
        }
      }
    }
    return propertyName;
  }

  /**
   * Get property name (all property names are cached for performance reasons
   * and indexed by their full name). If it isn't available, it will be created
   * automatically.
   *
   * @param namespace property namespace
   * @param localName property local name
   * @return propertyName
   * @exception AbstractPropertyException Exception raised in failure situation
   */
  public static IPropertyName getPropertyName(final String namespace, final String localName)
    throws AbstractPropertyException {
    // Construct full name
    String fullName = new StringBuffer("{")
      .append(namespace)
      .append("}")
      .append(localName)
      .toString();

    // Retrieve or create flyweight property name based on class and full name
    IPropertyName propertyName = (IPropertyName)getFlyWeight(PropertyName.class, fullName);
    if (propertyName == null) {
      // Check contraints
      try {
        checkConstraints(namespace, localName);
      }
      catch (IllegalArgumentException exception) {
        //$JL-EXC$ 
        throw new PropertyNameInvalidException(null, null, exception.getMessage());
      }

      // Create property name
      try {
        propertyName = new PropertyName(namespace, localName);
      }
      catch (IllegalArgumentException exception) {
        if ((propertyName = (IPropertyName)getFlyWeight(PropertyName.class, fullName)) == null) {
          throw exception;
        }
      }
    }
    return propertyName;
  }
}
