/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.property;

/**
 * Class implementing the internal system attributes of a property. They are
 * mapped into a bit array hold by the flags field in the property class for
 * greater performance and less memory consumption.
 *
 * @author Vedran Lerenc
 * @created 31. Januar 2003
 */
public final class PropertyFlags {
  /**
   * Bit mask for the internal system attribute named "required".
   */
  public final static int REQUIRED_BIT_MASK = 1 << 0;

  /**
   * Bit mask for the internal system attribute named "live".
   */
  public final static int LIVE_BIT_MASK = 1 << 1;

  /**
   * Bit mask for the internal system attribute named "read only".
   */
  public final static int READONLY_BIT_MASK = 1 << 2;

  /**
   * Bit mask for the internal system attribute named "hidden".
   */
  public final static int HIDDEN_BIT_MASK = 1 << 3;

  /**
   * Construct property flags based on given explicit internal system
   * attributes.
   *
   * @param required true if this property should be required
   * @param live true if this property should be live
   * @param readOnly true if this property should be read only
   * @param hidden true if this property should be hidden
   * @return property flags
   */
  public final static int constructFlags(
    boolean required,
    boolean live,
    boolean readOnly,
    boolean hidden) {
    return (required ? REQUIRED_BIT_MASK : 0)
       + (live ? LIVE_BIT_MASK : 0)
       + (readOnly ? READONLY_BIT_MASK : 0)
       + (hidden ? HIDDEN_BIT_MASK : 0);
  }

  /**
   * Returns true if these property flags indicate that its holding property has
   * the internal system attribute "required".
   *
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @return true if this property is required
   */
  public final static boolean isRequired(int flags) {
    return (flags & REQUIRED_BIT_MASK) != 0;
  }

  /**
   * Returns true if these property flags indicate that its holding property has
   * the internal system attribute "live".
   *
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @return true if this property is live
   */
  public final static boolean isLive(int flags) {
    return (flags & LIVE_BIT_MASK) != 0;
  }

  /**
   * Returns true if these property flags indicate that its holding property has
   * the internal system attribute "read only".
   *
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @return true if this property is read only
   */
  public final static boolean isReadOnly(int flags) {
    return (flags & READONLY_BIT_MASK) != 0;
  }

  /**
   * Returns true if these property flags indicate that its holding property has
   * the internal system attribute "hidden".
   *
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @return true if this property is hidden
   */
  public final static boolean isHidden(int flags) {
    return (flags & HIDDEN_BIT_MASK) != 0;
  }
}
