/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.property;

import com.sap.netweaver.bc.rf.common.exception.*;

import java.util.*;

/**
 * Read-Only interface describing a property, which is part of the resource and
 * describes certain aspects of it. A property has a name ({@link
 * IPropertyName}), type ({@link PropertyType}), some attributes, a natural
 * language description and most important a value. For each type exists a
 * specific property interface and implementation extending this interface.
 *
 * @author Vedran Lerenc
 * @created 29. Januar 2003
 * @see IPropertyName
 * @see PropertyType
 * @see IMutableProperty
 */
public interface IProperty {
  /**
   * Get immutable name of this property.
   *
   * @return immutable name of this property
   */
  public IPropertyName getPropertyName();

  /**
   * Get immutable type of this property.
   *
   * @return immutable type of this property
   */
  public PropertyType getType();

  /**
   * Return true if the internal system attributes are set.
   *
   * @return true if the internal system attributes are set
   */
  public boolean areFlagsSet();

  /**
   * Get bit array holding internal system attributes according to PropertyFlags
   * bit mask fields.
   *
   * @return bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   */
  public int getFlags();

  /**
   * Returns true if this property is a multi-value property.
   *
   * @return true if this property is a multi-value property
   */
  public boolean isMultiValue();

  /**
   * Returns true if this property is required.
   *
   * @return true if this property is required
   */
  public boolean isRequired();

  /**
   * Returns true if this property is live.
   *
   * @return true if this property is live
   */
  public boolean isLive();

  /**
   * Returns true if this property is read only.
   *
   * @return true if this property is read only
   */
  public boolean isReadOnly();

  /**
   * Returns true if this property is hidden.
   *
   * @return true if this property is hidden
   */
  public boolean isHidden();

  /**
   * Get description of property. Usually the description will be looked up in a
   * resource bundle using the property name as key. If no resource bundle is
   * provided, the property name may be returned.
   *
   * @return description of property
   */
  public String getDescription();

  /**
   * Get localized description of property. Usually the localized description
   * will be looked up in a resource bundle using the property name as key. If
   * no resource bundle is provided, the property name may be returned.
   *
   * @param locale locale to be used for localization
   * @return localized description of property
   */
  public String getDescription(Locale locale);

  /**
   * Get all attributes for the property at once. The returned attribute map
   * will be the stored one (as is - without being copied). Note that the stored
   * attribute map is no longer connected to the property when it is set by a
   * setAttributes() call.
   *
   * @return property map holding all attributes
   * @exception ResourceException when the property is invalid
   */
  public Properties getAttributes()
    throws ResourceException;

  /**
   * Get attribute for the property.
   *
   * @param attributeName attribute name
   * @return attribute value
   * @exception ResourceException when the attribute name or the property is
   *      invalid
   */
  public String getAttribute(String attributeName)
    throws ResourceException;

  /**
   * Get the property value converted to a string. If this is a multi value
   * property just return the first value.
   *
   * @return property value converted to a string
   * @exception ResourceException when the property is invalid
   */
  public String getValueAsString()
    throws ResourceException;

  /**
   * Get the property values converted to a string array. If this is a single
   * value property just return an array with the value as only element.
   *
   * @return array of property values converted to strings
   * @exception ResourceException when the property is invalid
   */
  public String[] getValuesAsStrings()
    throws ResourceException;
}
