/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.property;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.util.resource.*;

import com.sap.tc.logging.*;

import java.io.*;
import java.util.*;

/**
 * Class implementing a lean property (for less memory consumption), which is
 * part of the resource and describes certain aspects of it. A property has a
 * name ({@link IPropertyName}), type ({@link PropertyType}), some internal
 * system attributes ({@link PropertyFlags}), some more arbitrary attributes, a
 * natural language description and most important a value. For each type exists
 * a specific property interface and implementation extending this interface.
 *
 * @author Vedran Lerenc
 * @created 31. Januar 2003
 * @see IProperty
 * @see PropertyFlags
 */
public abstract class AbstractProperty
   implements IProperty, Serializable {
  // Location object for logging/tracing
  private static Location logger =
    Location.getLocation(com.sap.netweaver.bc.rf.common.property.AbstractProperty.class);

  // Property fields representing the complete internal state
  protected IPropertyName name;
  protected Properties attributes;
  protected int flags;
  protected ResourceBundles resourceBundles;

  /**
   * Construct property instance based on its internals.
   *
   * @param name property name
   * @param attributes property attributes
   * @param flags bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   * @param resourceBundles resource bundles containing (localized) property
   *      description
   * @exception ResourceException when the property creation failed
   */
  protected AbstractProperty(
    IPropertyName name,
    Properties attributes,
    int flags,
    ResourceBundles resourceBundles)
    throws ResourceException {
    this.name = name;
    this.attributes = attributes;
    this.flags = flags;
    if (resourceBundles == null) {
      this.resourceBundles = null;// TBD: System properties (usually comming with no resource bundles) should all use the same resource bundles
    }
    else {
      this.resourceBundles = resourceBundles;
    }
  }

  /**
   * Get immutable name of this property.
   *
   * @return immutable name of this property
   */
  public IPropertyName getPropertyName() {
    return name;
  }

  /**
   * Return true if the internal system attributes are set.
   *
   * @return true if the internal system attributes are set
   */
  public boolean areFlagsSet() {
    return flags != -1;
  }

  /**
   * Get bit array holding internal system attributes according to PropertyFlags
   * bit mask fields.
   *
   * @return bit array holding internal system attributes according to
   *      PropertyFlags bit mask fields
   */
  public int getFlags() {
    return flags;
  }

  /**
   * Returns true if this property is required.
   *
   * @return true if this property is required
   */
  public boolean isRequired() {
    return PropertyFlags.isRequired(flags);
  }

  /**
   * Returns true if this property is live.
   *
   * @return true if this property is live
   */
  public boolean isLive() {
    return PropertyFlags.isLive(flags);
  }

  /**
   * Returns true if this property is read only.
   *
   * @return true if this property is read only
   */
  public boolean isReadOnly() {
    return PropertyFlags.isReadOnly(flags);
  }

  /**
   * Returns true if this property is hidden.
   *
   * @return true if this property is hidden
   */
  public boolean isHidden() {
    return PropertyFlags.isHidden(flags);
  }

  /**
   * Get description of property. Usually the description will be looked up in a
   * resource bundle using the property name as key. If no resource bundle is
   * provided, the property name may be returned.
   *
   * @return description of property
   */
  public String getDescription() {
    String description = null;
    try {
      if ((resourceBundles != null)
         && (resourceBundles.exists(name.getFullName()))) {
        description =
          resourceBundles.getString(name.getFullName());
      }
    }
    catch (MissingResourceException exception) {
      // Do nothing; standard behaviour will return property name
      logger.warningT(
        "getDescription(163)",
        "Requested description on property named "
         + name.getFullName()
         + " but found no valid description despite a given resource bundle!");
    }
    if (description == null) {
      return name.getLocalName();
    }
    return description;
  }

  /**
   * Get localized description of property. Usually the localized description
   * will be looked up in a resource bundle using the property name as key. If
   * no resource bundle is provided, the property name may be returned.
   *
   * @param locale locale to be used for localization
   * @return localized description of property
   */
  public String getDescription(Locale locale) {
    String description = null;
    try {
      if ((resourceBundles != null)
         && (resourceBundles.exists(name.getFullName()))) {
        description =
          resourceBundles.getString(
          name.getFullName(),
          locale);
      }
    }
    catch (MissingResourceException exception) {
      // Do nothing; standard behaviour will return property name
      logger.warningT(
        "getDescription(196)",
        "Requested localized description ("
         + locale.getDisplayName()
         + ") on property named "
         + name.getFullName()
         + " but found no valid description despite a given resource bundle!");
    }
    if (description == null) {
      return name.getLocalName();
    }
    return description;
  }

  /**
   * Get all attributes for the property at once. The returned attribute map
   * will be the stored one (as is - without being copied). Note that the stored
   * attribute map is no longer connected to the property when it is set by a
   * setAttributes() call.
   *
   * @return property map holding all attributes
   * @exception ResourceException when the property is invalid
   */
  public Properties getAttributes()
    throws ResourceException {
    if (attributes == null) {
      attributes = new Properties();
    }
    return attributes;
  }

  /**
   * Get attribute for the property.
   *
   * @param attributeName attribute name
   * @return attribute value
   * @exception ResourceException when the attribute name or the property is
   *      invalid
   */
  public String getAttribute(String attributeName)
    throws ResourceException {
    if (attributes != null) {
      return attributes.getProperty(attributeName);
    }
    return null;
  }

  /**
   * Get hash code of this object. The hash code will be computed only based on
   * the property name.
   *
   * @return hash code of this object
   * @see java.lang.Object#hashCode()
   */
  public int hashCode() {
    return name.hashCode();
  }

  /**
   * Check this and another object for equality. The equality will be computed
   * only based on the property name.
   *
   * @param obj object to compare this object with
   * @return true if both objects are considered to be equal, false otherwise
   * @see java.lang.Object#equals(java.lang.Object)
   */
  public boolean equals(Object obj) {
    if (obj instanceof IProperty) {
      if (name.equals(((IProperty)obj).getPropertyName())) {
        return true;
      }
    }
    return false;
  }

  /**
   * Get string representation of this object.
   *
   * @return string representation of this object
   * @see java.lang.Object#toString()
   */
  public String toString() {
    StringBuffer result = new StringBuffer("Property ");
    result.append(name.getFullName());
    result.append(" ");
    result.append(isRequired() ? "[Required]" : "[Optional]");
    result.append(isReadOnly() ? "[ReadOnly]" : "[ReadWrite]");
    result.append(isHidden() ? "[Hidden]" : "[Visible]");
    result.append(" having ");
    result.append(
      (attributes == null) ? "no" : String.valueOf(attributes.size()));
    result.append(" inner attributes");
    return result.toString();
  }
}
