/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.namespace;

/**
 * Class describing an order position, that is used to position a child resource
 * below a resource collection.
 *
 * @created 21. Februar 2003
 */
public class OrderPosition {
  protected OrderPositionType type;
  protected String resourceHandleName;
  protected String referencePointName;

  /**
   * Private general constructor of an order position.
   *
   * @param type order position type of the resource to be reordered
   * @param resourceHandleName resource handle of the resource to be reordered
   * @param referencePointName reference point to which the resource should be
   *      aligned according to the order position
   */
  protected OrderPosition(OrderPositionType type, String resourceHandleName, String referencePointName) {
    this.type = type;
    this.resourceHandleName = resourceHandleName;
    this.referencePointName = referencePointName;
  }

  /**
   * Create order position to make the given resource the first resource of the
   * resource collection.
   *
   * @param resourceHandleName resource handle of the resource to be reordered
   * @return order position for use in a reorder call
   */
  public static OrderPosition createFirstPosition(String resourceHandleName) {
    return new OrderPosition(OrderPositionType.FIRST, resourceHandleName, null);
  }

  /**
   * Create order position to make the given resource the last resource of the
   * resource collection.
   *
   * @param resourceHandleName resource handle of the resource to be reordered
   * @return order position for use in a reorder call
   */
  public static OrderPosition createLastPosition(String resourceHandleName) {
    return new OrderPosition(OrderPositionType.LAST, resourceHandleName, null);
  }

  /**
   * Create order position to place the given resource before another resource
   * in the set of children of the resource collection.
   *
   * @param resourceHandleName resource handle of the resource to be reordered
   * @param referencePointName reference point to which the resource should be
   *      aligned according to the order position
   * @return order position for use in a reorder call
   */
  public static OrderPosition createBeforePosition(String resourceHandleName, String referencePointName) {
    return new OrderPosition(OrderPositionType.BEFORE, resourceHandleName, referencePointName);
  }

  /**
   * Create order position to place the given resource after another resource in
   * the set of children of the resource collection.
   *
   * @param resourceHandleName resource handle of the resource to be reordered
   * @param referencePointName reference point to which the resource should be
   *      aligned according to the order position
   * @return order position for use in a reorder call
   */
  public static OrderPosition createAfterPosition(String resourceHandleName, String referencePointName) {
    return new OrderPosition(OrderPositionType.AFTER, resourceHandleName, referencePointName);
  }

  /**
   * Get order position type of the resource to be reordered.
   *
   * @return order position type of the resource to be reordered
   */
  public OrderPositionType getType() {
    return type;
  }

  /**
   * Get resource handle of the resource to be reordered.
   *
   * @return resource handle of the resource to be reordered
   */
  public String getResourceHandleName() {
    return resourceHandleName;
  }

  /**
   * Get reference point to which the resource should be aligned according to
   * the order position.
   *
   * @return reference point to which the resource should be aligned according
   *      to the order position
   */
  public String getReferencePointName() {
    return referencePointName;
  }
}
