/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.namespace;

import com.sap.netweaver.bc.rf.util.namespace.IName;

/**
 * Constructs {@link IQueryExpression}s for later use with the corresponding
 * property search manager. <p>
 *
 * A IQueryBuilder constructs and combines IQueryExpressions, creating new
 * IQueryExpressions, which finally are executed against the property search
 * manager. Construction is done in the following way <pre>
 * builder.equ(x, y).and(builder.eq(z, "name")).or(builder.like(y, "%jpg"))
 *
 * builder.not(builder.greaterThan(x, new Integer(7)))
 * </pre> <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author stefan.eissing@greenbytes.de
 * @author julian.reschke@greenbytes.de
 * @created 14. April 2003
 * @version $Revision: #2 $
 */
public interface IQueryBuilder {

  /**
   * Returns a new expression which tests property == value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property == value
   */
  public IQueryExpression eq(IName name, Object value);

  /**
   * Returns a new expression which tests property == value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property == value
   */
  public IQueryExpression eq(IName name, String value);

  /**
   * Returns a new expression which tests property == value, ignoring case.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property == value
   */
  public IQueryExpression eqIgnoreCase(IName name, String value);


  /**
   * Returns a new expression which tests property != value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property != value
   */
  public IQueryExpression notEq(IName name, Object value);

  /**
   * Returns a new expression which tests property != value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property != value
   */
  public IQueryExpression notEq(IName name, String value);

  /**
   * Returns a new expression which tests property != value, ignoring case.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property != value
   */
  public IQueryExpression notEqIgnoreCase(IName name, String value);


  /**
   * Returns a new expression which tests property < value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property < value
   */
  public IQueryExpression less(IName name, Object value);

  /**
   * Returns a new expression which tests property < value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property < value
   */
  public IQueryExpression less(IName name, String value);

  /**
   * Returns a new expression which tests property < value, ignoring case.
   *
   * @param name the name3 of the property
   * @param value to compare against
   * @return expression representing property < value
   */
  public IQueryExpression lessIgnoreCase(IName name, String value);

  /**
   * Returns a new expression which tests property <= value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property <= value
   */
  public IQueryExpression lessEq(IName name, Object value);

  /**
   * Returns a new expression which tests property <= value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property <= value
   */
  public IQueryExpression lessEq(IName name, String value);

  /**
   * Returns a new expression which tests property <= value, ignoring case.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property <= value
   */
  public IQueryExpression lessEqIgnoreCase(IName name, String value);


  /**
   * Returns a new expression which tests property > value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property > value
   */
  public IQueryExpression greater(IName name, Object value);

  /**
   * Returns a new expression which tests property > value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property > value
   */
  public IQueryExpression greater(IName name, String value);

  /**
   * Returns a new expression which tests property > value, ignoring case.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property > value
   */
  public IQueryExpression greaterIgnoreCase(IName name, String value);


  /**
   * Returns a new expression which tests property >= value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property >= value
   */
  public IQueryExpression greaterEq(IName name, Object value);

  /**
   * Returns a new expression which tests property >= value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property >= value
   */
  public IQueryExpression greaterEq(IName name, String value);

  /**
   * Returns a new expression which tests property >= value, ignoring case.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property >= value
   */
  public IQueryExpression greaterEqIgnoreCase(IName name, String value);

  /**
   * Returns a new expression which is the complement of the given expression.
   *
   * @param expression to return complement of
   * @return expression representing !expression
   */
  public IQueryExpression not(IQueryExpression expression);

  /**
   * Returns a new expression which tests property is like value, where value is
   * a wildcard expression. Wildcard characters are "_" and "%", where "_"
   * matches one character and "%" matches 0 or more characters. "\" can be used
   * as escape character and is only allowed to escape itself, "_" and "%".
   *
   * @param name the name of the property
   * @param value the wildcard expression
   * @return expression representing property like value
   */
  public IQueryExpression like(IName name, String value);

  /**
   * Returns a new expression which tests property is like value, where value is
   * a wildcard expression. Wildcard characters are "_" and "%", where "_"
   * matches one character and "%" matches 0 or more characters. "\" can be used
   * as escape character and is only allowed to escape itself, "_" and "%".
   *
   * @param name the name of the property
   * @param value the wildcard expression
   * @return expression representing property like value
   */
  public IQueryExpression likeIgnoreCase(IName name, String value);

  /**
   * Returns a new expression which tests for the properties existence.
   *
   * @param name the name of the property
   * @return expression representing "if exists property"
   */
  public IQueryExpression isDefined(IName name);

  /**
   * Returns a new expression which tests if the resource is a collection
   *
   * @return expression representing test if resource is collection
   */
  public IQueryExpression isCollection();

  /**
   * @return constant false
   */
  public IQueryExpression constFalse();

  /**
   * @return constant true
   */
  public IQueryExpression constTrue();
}

