/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.namespace;

import java.util.*;

/**
 * Class implementing copy flags for a resource to be copied.
 *
 * @created 23. Januar 2003
 */
public class CopyFlagsDescriptor implements ICopyFlagsDescriptor {
  protected boolean childrenToBeCopied;
  protected Set childrenLinkTypesToBeCopied;

  /**
   * Construct object of class CopyFlagsDescriptor.
   *
   * @param childrenToBeCopied flag whether or not children should be copied
   */
  public CopyFlagsDescriptor(boolean childrenToBeCopied) {
    this.childrenToBeCopied = childrenToBeCopied;
    this.childrenLinkTypesToBeCopied = null;
  }

  /**
   * Construct object of class CopyFlagsDescriptor.
   *
   * @param childrenToBeCopied flag whether or not children should be copied
   * @param childrenLinkTypesToBeCopied flag whether or not child links of the
   *      given {@link LinkType} should be copied
   */
  public CopyFlagsDescriptor(
    boolean childrenToBeCopied,
    LinkType[] childrenLinkTypesToBeCopied) {
    this.childrenToBeCopied = childrenToBeCopied;
    if (childrenLinkTypesToBeCopied == null) {
      this.childrenLinkTypesToBeCopied = null;
    }
    else {
      this.childrenLinkTypesToBeCopied = new HashSet();
      for (int i = 0; i < childrenLinkTypesToBeCopied.length; i++) {
        this.childrenLinkTypesToBeCopied.add(
          childrenLinkTypesToBeCopied[i]);
      }
    }
  }

  /**
   * Add link type to be copied when children also need to be copied.
   *
   * @param childrenLinkTypeToBeCopied link type
   */
  public void addLinkTypeToBeCopied(LinkType childrenLinkTypeToBeCopied) {
    if (childrenLinkTypesToBeCopied == null) {
      childrenLinkTypesToBeCopied = new HashSet();
    }
    childrenLinkTypesToBeCopied.add(childrenLinkTypeToBeCopied);
  }

  /**
   * Remove link type to be copied when children also need to be copied.
   *
   * @param childrenLinkTypeToBeCopied link type
   */
  public void removeLinkTypeToBeCopied(LinkType childrenLinkTypeToBeCopied) {
    childrenLinkTypesToBeCopied.remove(childrenLinkTypeToBeCopied);
    if (childrenLinkTypesToBeCopied.size() == 0) {
      childrenLinkTypesToBeCopied = null;
    }
  }

  /**
   * Get flag whether or not children should be copied.
   *
   * @return flag whether or not children should be copied
   */
  public boolean isCildrenToBeCopied() {
    return childrenToBeCopied;
  }

  /**
   * Get flag whether or not child links of the given {@link LinkType} should be
   * copied.
   *
   * @param linkType link type
   * @return flag whether or not child links of the given {@link LinkType}
   *      should be copied
   */
  public boolean isChildrenLinkToBeCopied(LinkType linkType) {
    if (childrenLinkTypesToBeCopied != null) {
      if (childrenLinkTypesToBeCopied.contains(linkType)) {
        return true;
      }
      return false;
    }
    return true;
  }
}
