/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.lock;

import com.sap.netweaver.bc.rf.common.*;

import java.util.*;

/**
 * Interface describing lock information.
 *
 * @created 20. Mrz 2003
 */

public interface ILockInfo {
  /**
   * Get lock token. The lock token must be either an absolute URI (as defined
   * in <a href="http://www.ietf.org/rfc/rfc2396.txt">RFC2396</a> or a relative
   * URI reference (in which case the <a
   * href="http://greenbytes.de/tech/webdav/rfc2518.html#rfc.section.6.4">
   * opaquelocktoken</a> URI scheme is assumed).
   *
   * @return lock token
   */
  public String getLockToken();

  /**
   * Get resource id this lock info is valid for.
   *
   * @return resource id
   */
  public IRid getRid();

  /**
   * Get lock depth.
   *
   * @return {@link LockDepthType} of lock
   */
  public LockDepthType getDepth();

  /**
   * Get lock scope.
   *
   * @return {@link LockScopeType} of lock
   */
  public LockScopeType getScope();

  /**
   * Get lock kind.
   *
   * @return {@link LockKindType} of lock
   */
  public LockKindType getKind();

  /**
   * Get creation date.
   *
   * @return creation date
   */
  public Date getCreationDate();

  /**
   * Get lock expiration timestamp.
   *
   * @return lock expiration timestamp
   */
  public Date getExpirationDate();

  /**
   * Get time remaining until lock expires.
   *
   * @return time remaining until lock expires (or <code>-1</code> for
   *      "infinite").
   */
  public long getTimeRemaining();

  /**
   * Get lock timeout in milliseconds.
   *
   * @return lock timeout in milliseconds
   */
  public long getTimeout();

  /**
   * Get user id of lock owner.
   *
   * @return user id of lock owner
   */
  public String getOwner();

  /**
   * Get client supplied owner for the lock. A client can submit his own owner
   * information that should be returned upon lock discovery. If present, the
   * format must be the XML serialization of a <a
   * href="http://www.greenbytes.de/tech/webdav/rfc2518.html#ELEMENT_owner">
   * DAV:owner</a> element.
   *
   * @return client supplied owner for the lock
   */
  public String getClientSuppliedOwner();
}
