/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.exception;
import com.sap.netweaver.bc.rf.common.*;
import com.sap.netweaver.bc.rf.util.exception.*;

import java.util.*;

/**
 * Indicates that the repository (the service provided by a method call) is not
 * available at this time but may available later. Examples:<br>
 * - The repository has a fixed amount of available resources to process a
 * request (e.g. backend connections) and all resources are in use by other
 * threads.<br>
 * - The repository backend system is overloaded (e.g. a WebDAV Server that
 * reports HTTP Status 503). This exception may be used by a repository
 * implementation as an alternative to blocking the calling thread until enough
 * resources are avialable to process the request. The caller may use this
 * exception to retry the method call at a later time (see {@link
 * #getRetryAfter}).
 *
 * @created 14. April 2003
 */
public class ServiceNotAvailableException extends ResourceException {
  // Definition of default auto exception log behaviour,
  // used when no user behaviour is supplied during exception construction
  private final static boolean LOG_EXCEPTION_ON_CREATION = true;

  // Retry interval, can be <code>null</code> (see {@link #getRetryAfter}).
  private final Date retryAfterDate;

  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   */
  public ServiceNotAvailableException(IRid rid, Date retryAfterDate) {
    super(rid, LOG_EXCEPTION_ON_CREATION);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ServiceNotAvailableException(IRid rid, Date retryAfterDate, boolean logExceptionOnCreation) {
    super(rid, logExceptionOnCreation);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param rootCause throwable object which caused this exception
   */
  public ServiceNotAvailableException(IRid rid, Date retryAfterDate, Throwable rootCause) {
    super(rid, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ServiceNotAvailableException(IRid rid, Date retryAfterDate, Throwable rootCause, boolean logExceptionOnCreation) {
    super(rid, rootCause, logExceptionOnCreation);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param message not localized message
   */
  public ServiceNotAvailableException(IRid rid, Date retryAfterDate, String message) {
    super(rid, message, LOG_EXCEPTION_ON_CREATION);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   */
  public ServiceNotAvailableException(IRid rid, Date retryAfterDate, String message, Object[] args) {
    super(rid, message, args, LOG_EXCEPTION_ON_CREATION);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   */
  public ServiceNotAvailableException(IRid rid, Date retryAfterDate, String message, Throwable rootCause) {
    super(rid, message, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   */
  public ServiceNotAvailableException(IRid rid, Date retryAfterDate, String message, Object[] args, Throwable rootCause) {
    super(rid, message, args, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param message not localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ServiceNotAvailableException(IRid rid, Date retryAfterDate, String message, boolean logExceptionOnCreation) {
    super(rid, message, logExceptionOnCreation);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ServiceNotAvailableException(IRid rid, Date retryAfterDate,
    String message,
    Object[] args,
    boolean logExceptionOnCreation) {
    super(rid, message, args, logExceptionOnCreation);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ServiceNotAvailableException(IRid rid, Date retryAfterDate,
    String message,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, message, rootCause, logExceptionOnCreation);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ServiceNotAvailableException(IRid rid, Date retryAfterDate,
    String message,
    Object[] args,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, message, args, rootCause, logExceptionOnCreation);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   */
  protected ServiceNotAvailableException(IRid rid, Date retryAfterDate, ResourceAccessor resourceAccessor, String key) {
    super(rid, resourceAccessor, key, LOG_EXCEPTION_ON_CREATION);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   */
  protected ServiceNotAvailableException(IRid rid, Date retryAfterDate,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args) {
    super(rid, resourceAccessor, key, args, LOG_EXCEPTION_ON_CREATION);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param rootCause throwable object which caused this exception
   */
  protected ServiceNotAvailableException(IRid rid, Date retryAfterDate,
    ResourceAccessor resourceAccessor,
    String key,
    Throwable rootCause) {
    super(rid, resourceAccessor, key, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   */
  protected ServiceNotAvailableException(IRid rid, Date retryAfterDate,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    Throwable rootCause) {
    super(rid, resourceAccessor, key, args, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ServiceNotAvailableException(IRid rid, Date retryAfterDate,
    ResourceAccessor resourceAccessor,
    String key,
    boolean logExceptionOnCreation) {
    super(rid, resourceAccessor, key, logExceptionOnCreation);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ServiceNotAvailableException(IRid rid, Date retryAfterDate,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    boolean logExceptionOnCreation) {
    super(rid, resourceAccessor, key, args, logExceptionOnCreation);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ServiceNotAvailableException(IRid rid, Date retryAfterDate,
    ResourceAccessor resourceAccessor,
    String key,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, resourceAccessor, key, rootCause, logExceptionOnCreation);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ServiceNotAvailableException(IRid rid, Date retryAfterDate,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, resourceAccessor, key, args, rootCause, logExceptionOnCreation);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param message not localized message
   */
  protected ServiceNotAvailableException(IRid rid, Date retryAfterDate, ResourceMessage message) {
    super(rid, message, LOG_EXCEPTION_ON_CREATION);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   */
  protected ServiceNotAvailableException(IRid rid, Date retryAfterDate, ResourceMessage message, Throwable rootCause) {
    super(rid, message, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param message not localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ServiceNotAvailableException(IRid rid, Date retryAfterDate, ResourceMessage message, boolean logExceptionOnCreation) {
    super(rid, message, logExceptionOnCreation);
    this.retryAfterDate = retryAfterDate;
  }


  /**
   * Constructs a new exception.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param retryAfterDate retry interval, can be <code>null</code> (see {@link
   *      #getRetryAfter}).
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ServiceNotAvailableException(IRid rid, Date retryAfterDate,
    ResourceMessage message,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, message, rootCause, logExceptionOnCreation);
    this.retryAfterDate = retryAfterDate;
  }

  /**
   * Returns the time when retries of the operation are recommended. Repeating
   * the operation before this time, will most likely raise this exception
   * again. If the best time for retry is unknown, <code>null</code> is retuned.
   *
   * @return time of first recommended retry or <code>null</code> if not known
   */
  public Date getRetryAfterDate() {
    return this.retryAfterDate;
  }
}
