/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.exception;

import com.sap.netweaver.bc.rf.common.*;
import com.sap.netweaver.bc.rf.util.exception.*;

/**
 * Base class for all exception thrown by repository sub-manager methods. Raised
 * if the specified arguments are illegal or the method call failed for some
 * reason that has no dedicated sub-classed exception.
 *
 * @author m.breitenfelder@sapportals.com
 * @created 14. April 2003
 */
public class ResourceException extends RepositoryException {
  // Resource id of the resource this exception is associated with (if any)
  private IRid rid;

  // Definition of default auto exception log behaviour,
  // used when no user behaviour is supplied during exception construction
  private final static boolean LOG_EXCEPTION_ON_CREATION = true;


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   */
  public ResourceException(IRid rid) {
    super(LOG_EXCEPTION_ON_CREATION);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ResourceException(IRid rid, boolean logExceptionOnCreation) {
    super(logExceptionOnCreation);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param rootCause throwable object which caused this exception
   */
  public ResourceException(IRid rid, Throwable rootCause) {
    super(rootCause, LOG_EXCEPTION_ON_CREATION);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ResourceException(IRid rid, Throwable rootCause, boolean logExceptionOnCreation) {
    super(rootCause, logExceptionOnCreation);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param message not localized message
   */
  public ResourceException(IRid rid, String message) {
    super(message, LOG_EXCEPTION_ON_CREATION);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   */
  public ResourceException(IRid rid, String message, Object[] args) {
    super(message, args, LOG_EXCEPTION_ON_CREATION);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   */
  public ResourceException(IRid rid, String message, Throwable rootCause) {
    super(message, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   */
  public ResourceException(IRid rid, String message, Object[] args, Throwable rootCause) {
    super(message, args, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param message not localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ResourceException(IRid rid, String message, boolean logExceptionOnCreation) {
    super(message, logExceptionOnCreation);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ResourceException(IRid rid,
    String message,
    Object[] args,
    boolean logExceptionOnCreation) {
    super(message, args, logExceptionOnCreation);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ResourceException(IRid rid,
    String message,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(message, rootCause, logExceptionOnCreation);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ResourceException(IRid rid,
    String message,
    Object[] args,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(message, args, rootCause, logExceptionOnCreation);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   */
  protected ResourceException(IRid rid, ResourceAccessor resourceAccessor, String key) {
    super(resourceAccessor, key, LOG_EXCEPTION_ON_CREATION);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   */
  protected ResourceException(IRid rid,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args) {
    super(resourceAccessor, key, args, LOG_EXCEPTION_ON_CREATION);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param rootCause throwable object which caused this exception
   */
  protected ResourceException(IRid rid,
    ResourceAccessor resourceAccessor,
    String key,
    Throwable rootCause) {
    super(resourceAccessor, key, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   */
  protected ResourceException(IRid rid,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    Throwable rootCause) {
    super(resourceAccessor, key, args, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ResourceException(IRid rid,
    ResourceAccessor resourceAccessor,
    String key,
    boolean logExceptionOnCreation) {
    super(resourceAccessor, key, logExceptionOnCreation);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ResourceException(IRid rid,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    boolean logExceptionOnCreation) {
    super(resourceAccessor, key, args, logExceptionOnCreation);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ResourceException(IRid rid,
    ResourceAccessor resourceAccessor,
    String key,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(resourceAccessor, key, rootCause, logExceptionOnCreation);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ResourceException(IRid rid,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(resourceAccessor, key, args, rootCause, logExceptionOnCreation);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param message not localized message
   */
  protected ResourceException(IRid rid, ResourceMessage message) {
    super(message, LOG_EXCEPTION_ON_CREATION);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   */
  protected ResourceException(IRid rid, ResourceMessage message, Throwable rootCause) {
    super(message, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param message not localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ResourceException(IRid rid, ResourceMessage message, boolean logExceptionOnCreation) {
    super(message, logExceptionOnCreation);
    this.rid = rid;
  }


  /**
   * Construct object of class ResourceException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ResourceException(IRid rid,
    ResourceMessage message,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(message, rootCause, logExceptionOnCreation);
    this.rid = rid;
  }


  /**
   * Get resource id of the resource this exception is associated with or null
   * if none is associated with.
   *
   * @return resource id of the resource this exception is associated with
   */
  public IRid getRID() {
    return this.rid;
  }
}
