/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.exception;

import com.sap.netweaver.bc.rf.common.*;
import com.sap.netweaver.bc.rf.util.exception.*;

/**
 * Class implementing exception raised when an argument was filled with an
 * illegal value for an operation.
 *
 * @created 31. Januar 2003
 */
public class ArgumentNotValidException extends ResourceException {
  // Name of the argument that caused the exception
  private String argumentName = null;

  // Definition of default auto exception log behaviour,
  // used when no user behaviour is supplied during exception construction
  private final static boolean LOG_EXCEPTION_ON_CREATION = true;

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   */
  public ArgumentNotValidException(IRid rid, String argumentName) {
    super(rid, LOG_EXCEPTION_ON_CREATION);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ArgumentNotValidException(IRid rid, String argumentName, boolean logExceptionOnCreation) {
    super(rid, logExceptionOnCreation);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param rootCause throwable object which caused this exception
   */
  public ArgumentNotValidException(IRid rid, String argumentName, Throwable rootCause) {
    super(rid, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ArgumentNotValidException(IRid rid, String argumentName, Throwable rootCause, boolean logExceptionOnCreation) {
    super(rid, rootCause, logExceptionOnCreation);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param message not localized message
   */
  public ArgumentNotValidException(IRid rid, String argumentName, String message) {
    super(rid, message, LOG_EXCEPTION_ON_CREATION);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   */
  public ArgumentNotValidException(IRid rid, String argumentName, String message, Object[] args) {
    super(rid, message, args, LOG_EXCEPTION_ON_CREATION);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   */
  public ArgumentNotValidException(IRid rid, String argumentName, String message, Throwable rootCause) {
    super(rid, message, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   */
  public ArgumentNotValidException(IRid rid, String argumentName, String message, Object[] args, Throwable rootCause) {
    super(rid, message, args, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param message not localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ArgumentNotValidException(IRid rid, String argumentName, String message, boolean logExceptionOnCreation) {
    super(rid, message, logExceptionOnCreation);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ArgumentNotValidException(IRid rid, String argumentName,
    String message,
    Object[] args,
    boolean logExceptionOnCreation) {
    super(rid, message, args, logExceptionOnCreation);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ArgumentNotValidException(IRid rid, String argumentName,
    String message,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, message, rootCause, logExceptionOnCreation);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public ArgumentNotValidException(IRid rid, String argumentName,
    String message,
    Object[] args,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, message, args, rootCause, logExceptionOnCreation);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   */
  protected ArgumentNotValidException(IRid rid, String argumentName, ResourceAccessor resourceAccessor, String key) {
    super(rid, resourceAccessor, key, LOG_EXCEPTION_ON_CREATION);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   */
  protected ArgumentNotValidException(IRid rid, String argumentName,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args) {
    super(rid, resourceAccessor, key, args, LOG_EXCEPTION_ON_CREATION);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param rootCause throwable object which caused this exception
   */
  protected ArgumentNotValidException(IRid rid, String argumentName,
    ResourceAccessor resourceAccessor,
    String key,
    Throwable rootCause) {
    super(rid, resourceAccessor, key, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   */
  protected ArgumentNotValidException(IRid rid, String argumentName,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    Throwable rootCause) {
    super(rid, resourceAccessor, key, args, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ArgumentNotValidException(IRid rid, String argumentName,
    ResourceAccessor resourceAccessor,
    String key,
    boolean logExceptionOnCreation) {
    super(rid, resourceAccessor, key, logExceptionOnCreation);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ArgumentNotValidException(IRid rid, String argumentName,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    boolean logExceptionOnCreation) {
    super(rid, resourceAccessor, key, args, logExceptionOnCreation);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ArgumentNotValidException(IRid rid, String argumentName,
    ResourceAccessor resourceAccessor,
    String key,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, resourceAccessor, key, rootCause, logExceptionOnCreation);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ArgumentNotValidException(IRid rid, String argumentName,
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, resourceAccessor, key, args, rootCause, logExceptionOnCreation);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param message not localized message
   */
  protected ArgumentNotValidException(IRid rid, String argumentName, ResourceMessage message) {
    super(rid, message, LOG_EXCEPTION_ON_CREATION);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   */
  protected ArgumentNotValidException(IRid rid, String argumentName, ResourceMessage message, Throwable rootCause) {
    super(rid, message, rootCause, LOG_EXCEPTION_ON_CREATION);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param message not localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ArgumentNotValidException(IRid rid, String argumentName, ResourceMessage message, boolean logExceptionOnCreation) {
    super(rid, message, logExceptionOnCreation);
    this.argumentName = argumentName;
  }

  /**
   * Construct object of class ArgumentNotValidException.
   *
   * @param rid resource id of the resource this exception is associated with
   *      (if any)
   * @param argumentName name of the argument
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected ArgumentNotValidException(IRid rid, String argumentName,
    ResourceMessage message,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(rid, message, rootCause, logExceptionOnCreation);
    this.argumentName = argumentName;
  }

  /**
   * Get name of the argument that caused the exception.
   *
   * @return name of the argument that caused the exception
   */
  public String getArgumentName() {
    return this.argumentName;
  }
}
