/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.context;

import com.sap.netweaver.bc.rf.util.context.*;

/**
 * Context factory providing means to get and set the access context.
 *
 * @created 20. Januar 2003
 */
public class AccessContextFactory extends AbstractIndependentThreadContextFactory {
  private static AccessContextFactory inst = new AccessContextFactory();

  /**
   * Get singleton instance of this class.
   *
   * @return singleton instance of this class
   */
  public static AccessContextFactory getInstance() {
    return inst;
  }

  /**
   * Get actual context from stack of contexts.
   *
   * @return actual context on the context stack
   * @exception ContextUnavailableException when no (more) context is available
   *      on the context stack
   */
  public IAccessContext getContext()
    throws ContextUnavailableException {
    return (IAccessContext)inst.getTypeUnsafeContext();
  }

  /**
   * Get actual context from stack of contexts. Return null if none is availble
   * and don't throw a ContextUnavailableException.
   *
   * @return actual context on the context stack
   */
  public IAccessContext getContextOrNullIfUnavailable() {
    return (IAccessContext)inst.getTypeUnsafeContextOrNullIfUnavailable();
  }

  /**
   * Replace actual context with new one in stack of contexts. If no context is
   * stored, this one will be added.
   *
   * @param context new actual context replacing the old one on the stack
   */
  public void setContext(IAccessContext context) {
    inst.setTypeUnsafeContext(context);
  }

  /**
   * Remove actual context including all other stacked contexts completely.
   */
  public void removeContext() {
    inst.removeTypeUnsafeContext();
  }

  /**
   * Get and remove actual context from the stack of contexts.
   *
   * @return last actual context on the context stack
   * @exception ContextUnavailableException when no (more) context is available
   *      on the context stack
   */
  public IAccessContext popContext()
    throws ContextUnavailableException {
    return (IAccessContext)inst.popTypeUnsafeContext();
  }

  /**
   * Add new context to the top of the stack of contexts.
   *
   * @param context new actual context
   */
  public void pushContext(IAccessContext context) {
    inst.pushTypeUnsafeContext(context);
  }
}
