/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.context;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.util.logging.*;

import com.sap.security.api.*;

import com.sap.tc.logging.*;

import java.util.*;

/**
 * Class implementing the access context information the system needs to know,
 * when working on resources.
 *
 * @created 23. Januar 2003
 */
public class AccessContext implements IAccessContext, java.io.Serializable {
  private static Location logger =
    Location.getLocation(com.sap.netweaver.bc.rf.common.context.AccessContext.class);

  public static long DEFAULT_ALLOWED_STALE_DATA_TIMESPAN = 2 * 60 * 1000;

  /**
   * Current client identity operating on the resouce.
   */
  protected transient String clientId = null;

  /**
   * Current user operating on the resouce.
   */
  protected transient IUser user = null;

  /**
   * Locale of current user operating on the resouce.
   */
  protected Locale locale = null;

  /**
   * Time the repository data may not be older than.
   */
  protected long allowedStaleDataTimespan = DEFAULT_ALLOWED_STALE_DATA_TIMESPAN;

  /**
   * Client defined property map.
   */
  protected Map properties = null;

  /**
   * Construct instance just with the current user.
   *
   * @param clientId id of the current client operating on the resouce
   * @param user current user operating on the resouce
   * @param allowedStaleDataTimespan time the repository data may not be older
   *      than
   */
  public AccessContext(String clientId, IUser user, long allowedStaleDataTimespan) {
    this(clientId, user, null, null, allowedStaleDataTimespan);
  }

  /**
   * Construct instance just with the current user and client defined
   * properties. In case of no user, the anoymous user will be created.
   *
   * @param clientId id of the current client operating on the resouce
   * @param user current user operating on the resouce
   * @param allowedStaleDataTimespan time the repository data may not be older
   *      than
   * @param properties initial client defined properties to be used or null for
   *      no client defined properties
   */
  public AccessContext(String clientId, IUser user, Map properties, long allowedStaleDataTimespan) {
    this(clientId, user, null, properties, allowedStaleDataTimespan);
  }

  /**
   * Construct instance just with the current user and a locale to be used. In
   * case of no user, the anoymous user will be created.
   *
   * @param clientId id of the current client operating on the resouce
   * @param user current user operating on the resouce
   * @param locale locale to be associated or null to use user resp. default
   *      locale
   * @param allowedStaleDataTimespan time the repository data may not be older
   *      than
   */
  public AccessContext(String clientId, IUser user, Locale locale, long allowedStaleDataTimespan) {
    this(clientId, user, locale, null, allowedStaleDataTimespan);
  }

  /**
   * Construct instance with the current user, a locale to be used and client
   * defined properties. In case of no user, the anoymous user will be created.
   *
   * @param clientId id of the current client operating on the resouce
   * @param user current user operating on the resouce
   * @param locale locale to be associated or null to use user resp. default
   *      locale
   * @param properties initial client defined properties to be used or null for
   *      no client defined properties
   * @param allowedStaleDataTimespan time the repository data may not be older
   *      than
   */
  public AccessContext(String clientId, IUser user, Locale locale, Map properties, long allowedStaleDataTimespan) {
    // Set client id
    this.clientId = clientId;

    // Set user
    if (user == null) {
      // Use anonymous user
      try {
        this.user = UMFactory.getAnonymousUserFactory().getAnonymousUser();
      }
      catch (Exception exception) {
        //$JL-EXC$ 
        logger.errorT(
          "AccessContext(131)", "Cannot instantiate anonymous user: "
           + LoggingFormatter.extractCallstack(exception));
      }
    }
    else {
      this.user = user;
    }

    // Set locale
    if (locale == null) {
      // Use user locale if user present
      if (this.user != null) {
        this.locale = this.user.getLocale();
      }

      // Use default locale if locale still null
      if (this.locale == null) {
        this.locale = Locale.getDefault();
      }
    }
    else {
      this.locale = locale;
    }

    // Set properties
    this.properties = properties;

    // Set time the repository data may not be older than
    this.allowedStaleDataTimespan = allowedStaleDataTimespan;

    // Log creation info
    if (logger.beInfo()) {
      logger.debugT("AccessContext(163)", "User " + this.user.getDisplayName() + " [" + this.user.getUniqueID() + "] will be used for new context.");
      logger.debugT("AccessContext(164)", "Locale " + this.locale.getDisplayName() + " will be used for new context.");
      logger.debugT("AccessContext(165)", "Properties " + ((this.properties == null) ? "undefined" : "with " + this.properties.size() + " elements will be used ") + " for new context.");
    }

    // Check for not authenticated user
    if ((this.user != null) && (!UMFactory.getLogonAuthenticator().isAuthenticated(user))) {
      logger.warningT(
        "AccessContext(171)", "User "
         + this.user.getDisplayName() + " [" + this.user.getUniqueID() + "]"
         + " to be used for new context is not authenticated: "
         + LoggingFormatter.extractCallstack(
        new Exception()));
    }
  }

  /**
   * Get id of the current client operating on the resouce.
   *
   * @return id of the current client operating on the resouce
   */
  public String getClientId() {
    return this.clientId;
  }

  /**
   * Set id of the current client operating on the resouce.
   *
   * @param clientId id of the current client operating on the resouce
   */
  public void setClientId(String clientId) {
    this.clientId = clientId;
  }

  /**
   * Get current user operating on the resouce.
   *
   * @return current user operating on the resouce
   */
  public IUser getUser() {
    return this.user;
  }

  /**
   * Set current user operating on the resouce.
   *
   * @param user current user operating on the resouce
   */
  public void setUser(IUser user) {
    if (user == null) {
      logger.warningT(
        "setUser(214)", "User to be set in context is null: "
         + LoggingFormatter.extractCallstack(
        new Exception()));
    }
    else if (!UMFactory.getLogonAuthenticator().isAuthenticated(user)) {
      logger.warningT(
        "setUser(220)", "User "
         + this.user.getDisplayName() + " [" + this.user.getUniqueID() + "]"
         + " to be set in context is not authenticated: "
         + LoggingFormatter.extractCallstack(
        new Exception()));
    }
    this.user = user;
  }

  /**
   * Get locale of the current user.
   *
   * @return locale of the current user
   */
  public Locale getLocale() {
    return this.locale;
  }

  /**
   * Set locale of the current user.
   *
   * @param locale locale of the current user
   */
  public void setLocale(Locale locale) {
    this.locale = locale;
  }

  /**
   * Get time the repository data may not be older than.
   *
   * @return time the repository data may not be older than
   */
  public long getAllowedStaleDataTimespan() {
    return allowedStaleDataTimespan;
  }

  /**
   * Set time the repository data may not be older than.
   *
   * @param allowedStaleDataTimespan time the repository data may not be older
   *      than
   */
  public void setAllowedStaleDataTimespan(long allowedStaleDataTimespan) {
    this.allowedStaleDataTimespan = allowedStaleDataTimespan;
  }

  /**
   * Get a client defined property from the context.
   *
   * @param name name of the client defined property
   * @return value of the client defined property
   */
  public Object getClientDefinedProperty(String name) {
    if (this.properties == null || name == null) {
      return null;
    }
    return this.properties.get(name);
  }

  /**
   * Set a client defined property in the context.
   *
   * @param name name of the client defined property
   * @param value value of the client defined property
   * @exception RepositoryException Exception raised in failure situation
   */
  public synchronized void setClientDefinedProperty(String name, Object value)
    throws RepositoryException {
    if (name == null) {
      throw new RepositoryException("Parameter name is null");
    }
    if (name.length() == 0) {
      throw new RepositoryException("Parameter name is empty");
    }
    if (this.properties == null) {
      this.properties = new HashMap();
    }
    this.properties.put(name, value);
  }

  /**
   * Clone access context.
   *
   * @return cloned access context
   */
  public IAccessContext cloneContext() {
    return new AccessContext(clientId, user, locale, properties, allowedStaleDataTimespan);
  }
}
