/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common.content;

import com.sap.netweaver.bc.rf.common.exception.ResourceException;

import java.io.IOException;
import java.io.InputStream;

/**
 * Access to the content data of a resource. <p>
 *
 *
 *
 * @author m.breitenfelder@sapportals.com
 * @author julian.reschke@greenbytes.de
 * @created 14. April 2003
 */

public class Content implements IContent {
  private final InputStream inputStream;
  protected IContentMetadata contentMetadata;

  /**
   * Constructs Content object, for use when media subtype and encoding are
   * available as individual values.
   *
   * @param inputStream input stream
   * @param contentMetadata content metadata
   */
  public Content(InputStream inputStream, IContentMetadata contentMetadata) {
    this.inputStream = inputStream;
    this.contentMetadata = contentMetadata;
  }

  /**
   * Constructs Content object, for use when media subtype and encoding are
   * available as individual values.
   *
   * @param inputStream input stream
   * @param mediaType media type of content (trailing parameters are ignored)
   * @param contentLength length of content (use -1 when unknown)
   * @param encoding character encoding (use <code>null</code> when unknown)
   * @param language content language (use <code>null</code> when unknown)
   */
  public Content(InputStream inputStream, String mediaType, long contentLength, String encoding, String language) {
    this.inputStream = inputStream;
    this.contentMetadata = new ContentMetadata(contentLength, getMimeType(mediaType),
      encoding == null ? getEncoding(mediaType) : encoding, language, null, null, null);
  }

  /**
   * Constructs Content object, for use when encoding is not known separately
   * but may be present in media type string.
   *
   * @param inputStream input stream
   * @param mediaType media type of content (trailing parameters except encoding
   *      information is ignored)
   * @param contentLength length of content (use -1 when unknown)
   */
  public Content(InputStream inputStream, String mediaType, long contentLength) {
    this.inputStream = inputStream;
    this.contentMetadata = new ContentMetadata(contentLength, getMimeType(mediaType),
      getEncoding(mediaType), null, null, null, null);
  }

  /**
   * Get the inputStream attribute of the Content object.
   *
   * @return The inputStream value
   * @exception ResourceException Exception raised in failure situation
   */
  public InputStream getInputStream()
    throws ResourceException {
    return this.inputStream;
  }

  /**
   * Get the metadata attribute of the Content object.
   *
   * @return The metadata value
   * @exception ResourceException Exception raised in failure situation
   */
  public IContentMetadata getContentMetadata()
    throws ResourceException {
    return this.contentMetadata;
  }

  /**
   * Close content input stream.
   */
  public void close() {
    try {
      this.inputStream.close();
    }
    catch (IOException ex) {
      //$JL-EXC$ 
      // Ignore already closed input stream
      ;
    }
  }

  /**
   * Get the mimeType attribute of the Content object.
   *
   * @param contentType content type
   * @return mime type
   */
  private String getMimeType(String contentType) {

    if (contentType == null) {
      return null;
    }
    else {
      int pos = contentType.indexOf(";");
      if (pos < 0) {
        return contentType.trim();
      }
      else {
        return contentType.substring(0, pos).trim();
      }
    }
  }

  /**
   * Get the encoding attribute of the Content object.
   *
   * @param contentType content type
   * @return encoding
   */
  private String getEncoding(String contentType) {

    if (contentType == null) {
      return null;
    }
    else {
      int pos = contentType.indexOf(";");
      if (pos < 0) {
        return null;
      }
      else {
        String modifiers = contentType.substring(pos + 1);
        java.util.StringTokenizer st = new java.util.StringTokenizer(modifiers, ";");

        while (st.hasMoreTokens()) {

          String tok = st.nextToken().trim();

          if (tok.toLowerCase().startsWith("charset=")) {

            String val = tok.substring("charset=".length());
            int from = 0;
            int to = val.length();

            if (val.charAt(0) == '"' || val.charAt(0) == '\'') {
              from += 1;
            }

            if (val.charAt(to - 1) == '"' || val.charAt(to - 1) == '\'') {
              to -= 1;
            }

            return val.substring(from, to);
          }
        }

        return null;
      }
    }
  }
}
