/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common;

/**
 * A <a href="http://www.ietf.org/rfc/rfc2396.txt">RFC2396</a> reference
 * implementation. <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author julian.reschke@greenbytes.de
 * @author stefan.eissing@greenbytes.de
 * @created 20. Mrz 2003
 * @version $Id: //javabas/com.sapportals.wcm/50_COR/src/java/nemesis/common/com/sapportals/wcm/repository/common/UriReference.java#4
 *      $
 */
public class UriReference implements IUriReference {

  private final boolean m_isAbsolute;
  private final IAbsoluteUri m_uri;
  private final String m_scheme;
  private final String m_authority;
  private final String m_path;
  private final String m_query;
  private final String m_fragment;

  private final String m_full;

  /**
   * Construct new absolute uri reference.
   *
   * @param uri the absolute uri
   * @param fragmentIdentifier the fragment of the uri reference
   */
  public UriReference(IAbsoluteUri uri, String fragmentIdentifier) {
    m_isAbsolute = true;
    m_uri = uri;
    String scheme = null;
    String authority = null;
    String path = "";
    String query = null;
    if (m_uri instanceof IHierarchicalUri) {
      IHierarchicalUri huri = (IHierarchicalUri)m_uri;
      scheme = huri.getScheme();
      authority = huri.getAuthority();
      path = huri.getPath();
      query = huri.getQuery();
    }
    m_scheme = scheme;
    m_authority = authority;
    m_path = (path != null) ? path : "";
    m_query = query;
    m_fragment = (fragmentIdentifier != null) ? fragmentIdentifier : "";
    m_full = fullString();
  }

  /**
   * Construct new relative uri reference without authority and scheme.
   *
   * @param path the path of the reference or empty
   * @param fragment used in reference of null
   * @param query TBD: Description of the incoming method parameter
   */
  public UriReference(String path, String query, String fragment) {
    this(null, null, path, query, fragment);
  }

  /**
   * Construct new relative uri reference without scheme.
   *
   * @param authority of the relative uri reference
   * @param path the path of the reference or empty
   * @param fragment used in reference of null
   * @param query TBD: Description of the incoming method parameter
   */
  public UriReference(String authority, String path, String query, String fragment) {
    this(null, authority, path, query, fragment);
  }

  /**
   * Construct new relative uri reference with scheme. This constructor is used
   * for a degenerate case as described in RFC 2396, page 32. The construction
   * of relative uri references with scheme is discouraged. Note that either
   * scheme or authority must be null.
   *
   * @param scheme of the relative uri reference
   * @param authority of the relative uri reference
   * @param path the path of the reference or empty
   * @param fragment used in reference of null
   * @param query TBD: Description of the incoming method parameter
   * @exception IllegalArgumentException Exception raised in failure situation
   * @throws java.lang.ArgumentNotValidException if both scheme and authority
   *      are not null
   */
  public UriReference(String scheme, String authority, String path, String query, String fragment)
    throws IllegalArgumentException {
    if (scheme != null && authority != null) {
      throw new IllegalArgumentException("one of scheme and authority must be null");
    }
    m_isAbsolute = false;
    m_uri = null;
    m_scheme = (scheme != null) ? scheme.toLowerCase() : null;
    m_authority = authority;
    m_path = (path != null) ? path : "";
    m_query = query;
    m_fragment = (fragment != null) ? fragment : "";
    m_full = fullString();
  }

  /**
   * Construct object of class UriReference.
   *
   * @param reference TBD: Description of the incoming method parameter
   * @param fragmentIdentifier TBD: Description of the incoming method parameter
   */
  public UriReference(IUriReference reference, String fragmentIdentifier) {
    if (reference instanceof UriReference) {
      UriReference other = (UriReference)reference;
      m_isAbsolute = other.m_isAbsolute;
      m_uri = other.m_uri;
      m_scheme = other.m_scheme;
      m_authority = other.m_authority;
      m_path = other.m_path;
      m_query = other.m_query;
    }
    else {
      m_isAbsolute = reference.isAbsolute();
      m_uri = reference.getUri();
      m_scheme = reference.getScheme();
      m_authority = reference.getAuthority();
      m_path = reference.getPath();
      m_query = reference.getQuery();
    }
    m_fragment = (fragmentIdentifier != null) ? fragmentIdentifier : "";
    m_full = fullString();
  }

  /**
   * TBD: Description of the Method.
   *
   * @param path TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public IUriReference appendPath(String path) {
    if (m_isAbsolute) {
      if (m_uri instanceof IHierarchicalUri) {
        IHierarchicalUri huri = (IHierarchicalUri)m_uri;
        return new UriReference(huri.appendPath(path), m_fragment);
      }
      else {
        throw new UnsupportedOperationException("uri does not support queries: " + m_uri);
      }
    }
    else {
      if (m_path != null) {
        path = this.m_path + (path.startsWith("/") ? path : '/' + path);
      }
      return new UriReference(this.m_scheme, m_authority, path, this.m_query, this.m_fragment);
    }
  }

  /**
   * Set the query attribute of the UriReference object.
   *
   * @param query The new query value
   * @return TBD: Description of the outgoing return value
   */
  public IUriReference setQuery(String query) {
    if (m_isAbsolute) {
      if (m_uri instanceof IHierarchicalUri) {
        IHierarchicalUri huri = (IHierarchicalUri)m_uri;
        return new UriReference(huri.setQuery(query), m_fragment);
      }
      else {
        throw new UnsupportedOperationException("uri does not support queries: " + m_uri);
      }
    }
    else {
      return new UriReference(this.m_scheme, this.m_authority, this.m_path, query, this.m_fragment);
    }
  }

  public boolean equals(Object other) {
    if (other instanceof UriReference) {
      return equals((UriReference)other);
    }
    else if (other instanceof IUriReference) {
      return equals((IUriReference)other);
    }
    else {
      return false;
    }
  }

  public boolean equals(IUriReference other) {
    if (other == this) {
      return true;
    }
    if (other == null) {
      return false;
    }

    if (m_isAbsolute == other.isAbsolute()) {
      if (!m_fragment.equals(other.getFragmentIdentifier())) {
        return false;
      }

      if (m_isAbsolute) {
        return m_uri.equals(other.getUri());
      }
      else {
        return m_path.equals(other.getPath())
           && ((m_query != null && m_query.equals(other.getQuery())) || (m_query == null && other.getQuery() == null))
           && ((m_authority != null && m_authority.equals(other.getAuthority()))
           || (m_authority == null && other.getAuthority() == null))
           && ((m_scheme != null && m_scheme.equalsIgnoreCase(other.getScheme())) || (m_scheme == null && other.getScheme() == null));
      }
    }
    return false;
  }

  public boolean equals(UriReference other) {
    if (other == this) {
      return true;
    }
    if (other == null) {
      return false;
    }

    if (m_isAbsolute == other.m_isAbsolute) {

      if (m_isAbsolute) {
        if (!m_fragment.equals(other.m_fragment)) {
          return false;
        }
        return m_uri.equals(other.m_uri);
      }
      else {
        return m_full.equals(other.m_full);
      }
    }
    return false;
  }

  /**
   * TBD: Description of the Method.
   *
   * @return TBD: Description of the outgoing return value
   */
  public int hashCode() {
    return m_full.hashCode();
  }

  /**
   * TBD: Description of the Method.
   *
   * @return TBD: Description of the outgoing return value
   */
  public String toExternalForm() {
    return toString();
  }

  /**
   * TBD: Description of the Method.
   *
   * @return TBD: Description of the outgoing return value
   */
  public String toString() {
    return m_full;
  }

  /**
   * Get the fragment identifier (without '#') of the uri
   *
   * @return fragment identifier or emtpy String
   */
  public String getFragmentIdentifier() {
    return m_fragment;
  }

  /**
   * Get the scheme of the reference
   *
   * @return scheme of reference or null
   */
  public String getScheme() {
    return m_scheme;
  }

  /**
   * Get the authority of the reference
   *
   * @return authority of reference or null
   */
  public String getAuthority() {
    return m_authority;
  }

  /**
   * Get the path of the reference
   *
   * @return path of reference or empty string
   */
  public String getPath() {
    return m_path;
  }

  /**
   * Get the query of the reference
   *
   * @return query of reference or null
   */
  public String getQuery() {
    return m_query;
  }

  /**
   * Get the uri if the reference is absolute
   *
   * @return uri of reference or null
   */
  public IAbsoluteUri getUri() {
    return m_uri;
  }

  /**
   * Determine if the reference is absolute
   *
   * @return if reference is absolute
   */
  public boolean isAbsolute() {
    return m_isAbsolute;
  }

  /**
   * With base Uri, resolve this reference to an absolute reference. If this
   * reference is already absolute, this is the identity operation.
   *
   * @param base uri to resolve this reference with
   * @return absolute uri reference
   */
  public IUriReference resolveWith(IAbsoluteUri base) {
    if (m_isAbsolute) {
      return this;
    }
    else {
      return new UriReference(base.resolve(this), m_fragment);
    }
  }

  //------------------------- private -----------------------------------

  private String fullString() {
    StringBuffer sb = new StringBuffer(128);
    if (m_isAbsolute) {
      sb.append(m_uri);
    }
    else {
      if (m_scheme != null) {
        sb.append(m_scheme).append(':');
      }
      if (m_authority != null) {
        sb.append("//").append(m_authority);
      }
      sb.append(m_path);
      if (m_query != null) {
        sb.append('?').append(m_query);
      }
    }
    if (m_fragment.length() > 0) {
      sb.append('#').append(m_fragment);
    }
    return sb.toString();
  }
}
