/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common;

import com.sap.tc.logging.Location;

import java.io.ByteArrayOutputStream;
import java.io.UnsupportedEncodingException;
import java.util.*;

/**
 * Utility class for decoding/encoding URIs in HTTP requests. Handles escaping
 * of reserved as well as non-7bit characters. <p>
 *
 * Instances of this class are not threadsafe. The class methods are threadsafe,
 * however. <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: //javabas/com.sapportals.wcm/50_COR/src/java/nemesis/common/com/sapportals/wcm/repository/common/UriCodec.java#4
 *      $
 */
public class UriCodec {

  private final static Location log = Location.getLocation(com.sap.netweaver.bc.rf.common.UriCodec.class);

  private final static String[] escapeSequencePath = new String[]{
  // control 00 - 0f
    "%00", "%01", "%02", "%03", "%04", "%05", "%06", "%07", "%08", "%09", "%0a", "%0b", "%0c", "%0d", "%0e", "%0f",
  // control 10 - 1f
    "%10", "%11", "%12", "%13", "%14", "%15", "%16", "%17", "%18", "%19", "%1a", "%1b", "%1c", "%1d", "%1e", "%1f",
  // sp  !      "      #      $      %      &      '      (      )      *      +      ,      -      .      /
    "%20", null, "%22", "%23", "%24", "%25", "%26", null, null, null, null, "%2b", "%2c", null, null, null,
  // 0   1      2      3      4      5      6      7      8      9      :      ;      <      =      >      ?
    null, null, null, null, null, null, null, null, null, null, "%3a", "%3b", "%3c", "%3d", "%3e", "%3f",
  // @   A      B      C      D      E      F      G      H      I      J      K      L      M      N      O
    "%40", null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
  // P   Q      R      S      T      U      V      W      X      Y      Z      [      \      ]      ^      _
    null, null, null, null, null, null, null, null, null, null, null, null, "%5c", null, "%5e", null,
  // `   a      b      c      d      e      f      g      h      i      j      k      l      m      n      o
    "%60", null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
  // p   q      r      s      t      u      v      w      x      y      z      {      |      }      ~      7f
    null, null, null, null, null, null, null, null, null, null, null, "%7b", "%7c", "%7d", null, "%7f",
    };

  private final static String[] escapeSequenceQuery = new String[]{
  // control 00 - 0f
    "%00", "%01", "%02", "%03", "%04", "%05", "%06", "%07", "%08", "%09", "%0a", "%0b", "%0c", "%0d", "%0e", "%0f",
  // control 10 - 1f
    "%10", "%11", "%12", "%13", "%14", "%15", "%16", "%17", "%18", "%19", "%1a", "%1b", "%1c", "%1d", "%1e", "%1f",
  // sp  !      "      #      $      %      &      '      (      )      *      +      ,      -      .      /
    "+", null, "%22", "%23", "%24", "%25", "%26", null, null, null, null, "%2b", null, null, null, null,
  // 0   1      2      3      4      5      6      7      8      9      :      ;      <      =      >      ?
    null, null, null, null, null, null, null, null, null, null, "%3a", "%3b", "%3c", "%3d", "%3e", "%3f",
  // @   A      B      C      D      E      F      G      H      I      J      K      L      M      N      O
    "%40", null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
  // P   Q      R      S      T      U      V      W      X      Y      Z      [      \      ]      ^      _
    null, null, null, null, null, null, null, null, null, null, null, null, "%5c", null, "%5e", null,
  // `   a      b      c      d      e      f      g      h      i      j      k      l      m      n      o
    "%60", null, null, null, null, null, null, null, null, null, null, null, null, null, null, null,
  // p   q      r      s      t      u      v      w      x      y      z      {      |      }      ~      7f
    null, null, null, null, null, null, null, null, null, null, null, "%7b", "%7c", "%7d", null, "%7f",
    };

  private final static String[] encodedBytes;

  static {
    String tmp[] = new String[256];
    for (int i = 0; i < tmp.length; i++) {
      String hex = Integer.toHexString(i);
      tmp[i] = "%" + (hex.length() == 1 ? "0" : "") + hex;
    }

    encodedBytes = tmp;
  }

  private ByteArrayOutputStream m_bos;
  private StringBuffer m_sb;

  /**
   * Encode the given URI. Escapes reserved URI characters.
   *
   * @param unquotedURI non-encoded URI
   * @return encoded URI
   */
  public static String Encode(String unquotedURI) {
    return encodeInternal(unquotedURI, new StringBuffer(1024), false, true);
  }

  /**
   * Decode the given URI. De-escapes reserved URI characters.
   *
   * @param quotedURI encoded URI
   * @return decoded URI
   */
  public static String Decode(String quotedURI) {
    return decodeInternal(quotedURI, new StringBuffer(1024), new ByteArrayOutputStream(1024));
  }

  /**
   * Encode the given http query string. Escapes reserved URI characters.
   *
   * @param params parameter to encode
   * @return encoded query string
   */
  public static String EncodeQuery(Properties params) {
    StringBuffer sb = new StringBuffer(128);
    StringBuffer scratch = new StringBuffer(128);

    Iterator iter = params.keySet().iterator();
    while (iter.hasNext()) {
      String key = (String)iter.next();
      String value = params.getProperty(key);

      if (sb.length() > 0) {
        sb.append('&');
      }

      if (key != null) {
        sb.append(encodeInternal(key, scratch, false, false));
        sb.append('=');
        if (value != null) {
          sb.append(encodeInternal(value, scratch, true, false));
        }
      }
    }
    return sb.toString();
  }

  /**
   * Encode the given http path string. Escapes reserved URI characters.
   *
   * @param path TBD: Description of the incoming method parameter
   * @return encoded query string
   */
  public static String EncodePath(String path) {
    StringBuffer scratch = new StringBuffer(128);
    return encodeInternal(path, scratch, false, false);
  }

  /**
   * Decode the given http query string. De-escapes reserved query characters.
   *
   * @param query http uri encoded
   * @return set of decoded parameters
   */
  public static Properties DecodeQuery(String query) {
    StringBuffer scratch = new StringBuffer(128);
    ByteArrayOutputStream bos = new ByteArrayOutputStream(128);
    Properties props = new Properties();

    StringTokenizer tok = new StringTokenizer(query, "&");
    while (tok.hasMoreTokens()) {
      String token = tok.nextToken();
      String key = token;
      String value = "";
      int index = token.indexOf('=');
      if (index >= 0) {
        key = token.substring(0, index);
        if (index < token.length()) {
          value = token.substring(index + 1).replace('+', ' ');
        }
      }
      key = decodeInternal(key, scratch, bos);
      value = decodeInternal(value, scratch, bos);
      props.setProperty(key, value);
    }
    return props;
  }

  /**
   * Remove a parameter from a query string if present. Returns the string
   * unchanged if parameter is not present.
   *
   * @param query to remove parameter from
   * @param paramName name of parameter
   * @return query string with parameter removed
   */
  public static String RemoveParameter(String query, String paramName) {
    boolean keyFound = false;
    StringBuffer result = new StringBuffer(query.length());

    StringTokenizer tok = new StringTokenizer(query, "&");
    while (tok.hasMoreTokens()) {
      String token = tok.nextToken();
      String key = token;
      String value = null;
      int index = token.indexOf('=');
      if (index >= 0) {
        key = token.substring(0, index);
        if (index + 1 < token.length()) {
          value = token.substring(index + 1);
        }
        else {
          value = "";
        }
      }
      if (key.equals(paramName)) {
        keyFound = true;
      }
      else {
        if (result.length() > 0) {
          result.append('&');
        }

        result.append(key);
        if (value != null) {
          result.append('=').append(value);
        }
      }
    }

    if (keyFound) {
      return result.toString();
    }
    return query;
  }

  /**
   * Creates a new URICoded Instance. It is safe to use this instance in one,
   * single thread.
   */
  public UriCodec() {
    m_bos = new ByteArrayOutputStream(1024);
    m_sb = new StringBuffer(1024);
  }

  /**
   * Converts an encoded URI into a Java String. Escaped octets are treated as
   * octets in UTF-8 encoding. <p>
   *
   * This method is not multithread-safe.
   *
   * @param quoted the quoted URI
   * @return the URI without escape characters
   */
  public String decode(String quoted) {
    return decodeInternal(quoted, m_sb, m_bos);
  }

  /**
   * Converts a URI in plain Java String to a format suitable for transmitting
   * in Http requests. First, it escapes the reserved characters in URIs.
   * Second, all characters not representable as 7-bit US-ASCII are escaped as
   * octets in their UTF-8 encoding. Regards the last question mark in the
   * string as query separator. <p>
   *
   * This method is not multithread-safe.
   *
   * @param unquoted URI in unquoted form
   * @return URI with escaped characters
   */
  public String encode(String unquoted) {
    return encodeInternal(unquoted, m_sb, false, true);
  }

  /**
   * Converts a URI in plain Java String to a format suitable for transmitting
   * in Http requests. First, it escapes the reserved characters in URIs.
   * Second, all characters not representable as 7-bit US-ASCII are escaped as
   * octets in their UTF-8 encoding. Does no special handling of question marks.
   * <p>
   *
   * This method is not multithread-safe.
   *
   * @param unquoted URI in unquoted form
   * @return URI with escaped characters
   */
  public String encodePath(String unquoted) {
    return encodeInternal(unquoted, m_sb, false, false);
  }

  // -------------------------- package ----------------------------------

  /**
   * Decode the given http query string. De-escapes reserved query characters.
   *
   * @param query http uri encoded
   * @return List of decoded parameters as {@link UriQuery.Parameter}s
   */
  static List DecodeQueryParameter(String query) {
    List result = new ArrayList(10);
    if (query == null) {
      return result;
    }

    StringBuffer scratch = new StringBuffer(128);
    ByteArrayOutputStream bos = new ByteArrayOutputStream(128);

    StringTokenizer tok = new StringTokenizer(query, "&");
    while (tok.hasMoreTokens()) {
      String token = tok.nextToken();
      String key = token;
      String value = "";
      int index = token.indexOf('=');
      if (index >= 0) {
        key = token.substring(0, index);
        if (index < token.length()) {
          value = token.substring(index + 1).replace('+', ' ');
        }
      }
      key = decodeInternal(key, scratch, bos);
      value = decodeInternal(value, scratch, bos);
      result.add(new UriQuery.Parameter(key, value));
    }
    return result;
  }

  /**
   * Encode the given http query string. Escapes reserved URI characters.
   *
   * @param params List of {@link UriQuery.Parameter}s
   * @return encoded query string
   */
  static String EncodeQueryParameter(List params) {
    StringBuffer sb = new StringBuffer(128);
    StringBuffer scratch = new StringBuffer(128);

    for (int i = 0, n = params.size(); i < n; ++i) {
      UriQuery.Parameter param = (UriQuery.Parameter)params.get(i);

      if (sb.length() > 0) {
        sb.append('&');
      }

      if (param.name != null) {
        sb.append(encodeInternal(param.name, scratch, false, false));
        if (param.value != null) {
          sb.append('=');
          sb.append(encodeInternal(param.value, scratch, true, false));
        }
      }
    }
    return sb.toString();
  }

  // -------------------------- private ------------------------------------

  private static String decodeInternal(String quoted,
    StringBuffer sb,
    ByteArrayOutputStream bos) {
    // Optimize for the common case...
    //
    if ((quoted.indexOf('%') < 0)) {
      return quoted;
    }

    String query = null;
    int index = quoted.lastIndexOf('?');
    if (index != -1) {
      query = quoted.substring(index);
      quoted = quoted.substring(0, index);
    }

    sb.setLength(0);

    int len = quoted.length();
    bos.reset();

    for (int i = 0; i < len; ++i) {
      char c = quoted.charAt(i);
      int octet = 0;
      if (c == '%' && len >= i + 2) {
        int hnibble = Character.digit(quoted.charAt(i + 1), 16);
        int lnibble = Character.digit(quoted.charAt(i + 2), 16);
        if (hnibble >= 0 && lnibble >= 0) {
          octet = (hnibble << 4) | lnibble;
          i += 2;
          c = (char)octet;
        }
      }

      if (octet < 128) {
        if (bos.size() > 0) {
          sb.append(bosToString(bos));
        }
        // normal char or quoted reserved character
        sb.append(c);
      }
      else {
        // quoted non 7bit character
        // there can be more than one encoding here
        bos.write(octet);
      }
    }
    if (bos.size() > 0) {
      sb.append(bosToString(bos));
    }

    if (query != null) {
      sb.append(query);
    }

    return sb.toString();
  }

  private static String encodeInternal(String unquoted, StringBuffer sb,
    boolean queryvalue, boolean handleqmark) {
    final String[] es = queryvalue ? escapeSequenceQuery : escapeSequencePath;
    try {
      sb.setLength(0);
      String query = null;
      if (handleqmark) {
        int index = unquoted.lastIndexOf('?');
        if (index != -1) {
          query = unquoted.substring(index);
          unquoted = unquoted.substring(0, index);
        }
      }

      byte[] bytes = unquoted.getBytes("UTF-8");
      for (int i = 0; i < bytes.length; i++) {
        byte b = bytes[i];
        if (b < 32) {
          // bit 8 set or control character
          sb.append(getURIEncoding(b));
        }
        else {
          String escape = es[b];
          if (escape != null) {
            sb.append(escape);
          }
          else {
            sb.append((char)b);
          }
        }
      }

      if (query != null) {
        sb.append(query);
      }

      return sb.toString();
    }
    catch (Exception ex) {
      //$JL-EXC$ 
      //ex.printStackTrace(System.err);
      return unquoted;
    }
  }

  private static String bosToString(ByteArrayOutputStream bos) {
    byte[] bytes = bos.toByteArray();
    bos.reset();
    if (bytes.length == 0) {
      return "";
    }

    try {
      if (looksLikeUTF8(bytes)) {
        return new String(bytes, "UTF-8");
      }
      else {
        return new String(bytes, "8859_1");
      }
    }
    catch (UnsupportedEncodingException ex) {
      //$JL-EXC$ 
      log.errorT("bosToString(471)", "converting bytes to string" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
      return new String(bytes);
    }
  }

  private final static int[] seqLen = new int[]
    {
  //0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, // 0
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, // 1
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, // 2
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, // 3
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, // 4
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, // 5
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, // 6
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, // 7
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, // 8
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, // 9
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, // a
  -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, // b
  -1, -1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // c
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, // d
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, // e
  3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 5, 5, -1, -1,// f
  };

  /**
   * Determines if bytes are a valid UTF-8 sequence.
   *
   * @param bytes TBD: Description of the incoming method parameter
   * @return if bytes are a valid UTF-8 sequence
   */
  public final static boolean looksLikeUTF8(byte[] bytes) {
    int sequenceLen = 0;
    for (int i = 0; i < bytes.length; ++i) {
      byte b = bytes[i];
      if (b < 0) {
        sequenceLen = getUTF8SequenceLength(b);
        if (sequenceLen < 0) {
          return false;
        }
        while (sequenceLen > 0) {
          --sequenceLen;
          ++i;
          if (i >= bytes.length) {
            return false;
          }
          b = bytes[i];
          if ((b & 0xc0) != 0x80) {
            return false;
          }
        }
      }
    }

    return true;
  }

  /**
   * Return the sequence length of a UTF-8 sequence starting with the given
   * byte. Returns -1 if byte is no valid sequence start.
   *
   * @param b byte for sequence start
   * @return length of UTF-8 sequence or -1
   */
  public final static int getUTF8SequenceLength(byte b) {
    return seqLen[b & 0x00ff];
  }

  /**
   * Get string with URI escapes for given byte.
   *
   * @param b byte to escape
   * @return string with uri encoding of byte
   */
  public final static String getURIEncoding(byte b) {
    return encodedBytes[b & 0x00ff];
  }
}
