/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common;

import com.sapportals.wcm.WcmException;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;

/**
 * A Rid is an identifier for a CM resource. It holds all information to
 * identify a CM resource and retrieve it via the CM Framework. Each CM resource
 * has its unique Rid. The first pathsegment must be a valid name of a
 * repository (prefix). <p>
 *
 * A Rid has a path-like structure, like a path in a file system and follows the
 * following production: <pre>
 *   Rid         ::= '/' pathsegment | '/' pathsegment '?' query
 *   pathsegment ::= name | name '/' pathsegment | empty
 *   name        ::= any char except '/'
 *   query       ::= parameter | parameter '&' query | empty
 *   parameter   ::= varname | varname '=' value
 *   varname     ::= any char except '=', '&'
 *   value       ::= any character except '&'
 * </pre> Equality is defined as ignoring trailing '/' characters. Thus, the
 * Rids "/docs/" and /docs" are considered equal. There is special handling for
 * the root collection "/", which is not equal to the empty Rid "". <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author frank.renkes@sapportals.com
 * @author stefan.eissing@greenbytes.de
 * @author m.breitenfelder@sapportals.com
 */
public final class Rid implements Comparable, Serializable, IRid {

  public final static char PATH_SEPARATOR = '/';

  private static Rid emptyRid = Rid.getRid("");
  private static Rid rootRid = Rid.getRid(new String(new char[]{Rid.PATH_SEPARATOR}));

  private String uri;
  private final String path;
  private final int pathlen;
  private final String query;
  private final boolean endsWithSlash;
  private final boolean isAbsolute;
  private int hashCode;
  private Rid root;
  private Rid name;
  private Rid parent;

// stefan.eissing@greenbytes.de: static caching of Rids disabled for now
// Problems that need to be addressed:
// - needs synchronization for read *and* write accesses
// - SoftReferences make garbage collection more difficult and less efficient
// - Rid("a?", null) and Rid("a?") have same key, but different external form
//
//  private static HashMap ridCache;

  public static IRid getRid(String path, String query) {
    return new Rid(path, query);
//    if (null == ridCache) ridCache = new HashMap();
//
//    String key;
//
//    if (query != null && query.length() > 0) {
//      key = path + "?" + query;
//    } else {
//      key = path;
//    }
//
//    RidReference ridRef = (RidReference)ridCache.get(key);
//    Rid      rid;
//
//    if (ridRef != null) {
//      if (null != (rid = ridRef.getRid())) {
//        return rid;
//      }
//    }
//
//    Rid newRid = new Rid(path, query);
//
//    ridCache.put(key, new RidReference(newRid));
//
//    return newRid;
  }

  public static Rid getRid(String uri) {
    return new Rid(uri);
//    if (null == ridCache) ridCache = new HashMap();
//
//    RidReference ridRef = (RidReference)ridCache.get(uri);
//    Rid      rid;
//
//    if (ridRef != null) {
//      if (null != (rid = ridRef.getRid())) {
//        return rid;
//      }
//    }
//
//    Rid newRid = new Rid(uri);
//
//    ridCache.put(uri, new RidReference(newRid));
//
//    return newRid;
  }


  /**
   * Create a new Rid with given path and query string
   *
   * @param path of Rid
   * @param query part or Rid (can be null)
   * @deprecated as of NW04. Use Rid.getRid(path, query)
   */
  public Rid(String path, String query) {
    this.path = (path == null) ? "" : path;
    this.pathlen = this.path.length();
    if (this.pathlen > 0) {
      this.endsWithSlash = this.path.charAt(this.pathlen - 1) == Rid.PATH_SEPARATOR;
      this.isAbsolute = this.path.charAt(0) == Rid.PATH_SEPARATOR;
    }
    else {
      this.endsWithSlash = false;
      this.isAbsolute = false;
    }
    this.query = query;
    if (this.query == null) {
      this.uri = this.path;
    }
  }

  /**
   * Create a new Rid from a string (possibly including a query part). <p>
   *
   *
   *
   * @param uri TBD: Description of the incoming method parameter
   * @deprecated as of NW04. Use Rid.getRid(uri)
   */
  public Rid(String uri) {
    // sei: I'd like to remove the recognition of the query string
    // here. However in the repository framework, the ResourceFactory
    // round-trips Rid and Rid.toString()
    // So:              Rid("/test", "a=b")
    // toString:        "/test?a=b"
    // new Rid(string); Rid("/test?a=b", null)
    // but correct is   Rid("/test", "a=b")
    //
    String path = (uri != null) ? uri : "";
    this.uri = path;
    String query = null;
    if (uri != null) {
      int index = path.lastIndexOf('?');
      if (index >= 0) {
        query = path.substring(index + 1);
        path = path.substring(0, index);
      }
    }
    this.path = path;
    this.pathlen = this.path.length();
    if (this.pathlen > 0) {
      this.endsWithSlash = this.path.charAt(this.pathlen - 1) == Rid.PATH_SEPARATOR;
      this.isAbsolute = this.path.charAt(0) == Rid.PATH_SEPARATOR;
    }
    else {
      this.endsWithSlash = false;
      this.isAbsolute = false;
    }
    this.query = query;
  }

  /**
   * @return TBD: Description of the outgoing return value
   * @deprecated as of NW04. Rid objects are never encoded
   */
  public Rid encode() {
    return Rid.getRid(UriCodec.Encode(this.path));
  }

  /**
   * @return TBD: Description of the outgoing return value
   * @deprecated as of NW04. Rid objects are never decoded
   */
  public Rid decode() {
    return Rid.getRid(UriCodec.Decode(this.path));
  }

  /**
   * Convert this Rid to a representation usable in RFC2396 URI references.
   * Non-ASCII and reserved characters are encoded.<p>
   *
   * Note that such an encoded string cannot be used to construct an new Rid
   * object! The path in Rid objects is never, ever encoded.
   *
   * @return Rid suitably escaped as RFC2396 URI reference
   */
  private String toExternalFormCache = null;

  public String toExternalForm() {
    if (toExternalFormCache == null) {
      if (this.query != null) {
        StringBuffer sb = new StringBuffer(256);
        sb.append(UriCodec.EncodePath(this.path));
        sb.append('?');
        sb.append(this.query);
        toExternalFormCache = sb.toString();
      }
      else {
        toExternalFormCache = UriCodec.EncodePath(this.path);
      }
    }
    return toExternalFormCache;
  }

  /**
   * Return the path component of this Rid.
   *
   * @return the path component
   */
  public String getPath() {
    return this.path;
  }

  /**
   * Return the query component of this Rid (null if it does not exist).
   *
   * @return the query component
   */
  public String getQuery() {
    return this.query;
  }

  /**
   * Get the query parameters of the Rid. Will return an empty Properties object
   * if Rid has no query part. Modifying the {@link Properties} object has no
   * effect on the Rid itself.
   *
   * @return query parameters as {@link Properties}
   */
  public Properties getQueryParameter() {
    if (this.query != null && this.query.length() > 0) {
      return UriCodec.DecodeQuery(this.query);
    }
    else {
      return new Properties();
    }
  }

  /**
   * Get Rid of toplevel collection (below root) of this Rid. <p>
   *
   * The root Rid of a toplevel collection is the root collection Rid. The root
   * Rid of the root Rid is the root Rid itself.
   *
   * @return Rid of toplevel collection below root
   */
  public IRid root() {
    if (null != this.root) {
      return this.root;
    }
    if (0 == this.pathlen) {
      return this;
    }

    int pos = this.path.indexOf(Rid.PATH_SEPARATOR, 1);
    this.root = (pos != -1) ? Rid.getRid(this.path.substring(0, pos)) : rootRid;

    return this.root;
  }

  /**
   * Get the name of the resource, designated by this Rid. If the Rid has query
   * parameter, these are discarded.
   *
   * @return the name of the resource, designated by this Rid
   */
  public IRid name() {
    if (null != this.name) {
      return this.name;
    }
    if (0 == this.pathlen) {
      return this;
    }

    if (this.endsWithSlash) {
      int pos = this.path.lastIndexOf(Rid.PATH_SEPARATOR, this.pathlen - 2);
      this.name = (Rid)Rid.getRid(this.path.substring(pos + 1, this.pathlen - 1), null);
    }
    else {
      int pos = this.path.lastIndexOf(Rid.PATH_SEPARATOR);
      this.name = (pos != -1) ? (Rid)Rid.getRid(this.path.substring(pos + 1), null) : this;
    }

    return this.name;
  }


  /**
   * Get the Rid of the parent collection of this Rid
   *
   * @return the Rid of the parent collection
   */
  public IRid parent() {
    if (null != this.parent) {
      return this.parent;
    }
    this.parent = (Rid)removeName();
    return this.parent;
  }

  /**
   * Get the extension part of the resource name. Returns the empty string if
   * there is no extension.
   *
   * @return the extension part of the resource name
   */
  public String extension() {
    if (this.pathlen == 0) {
      return "";
    }

    Rid n = (Rid)name();
    int pos = n.path.lastIndexOf('.');
    if (pos != -1 && pos < n.pathlen - 1) {
      return n.path.substring(pos + 1);
    }
    else {
      return "";
    }
  }

  /**
   * Get a new Rid with the toplevel collection removed
   *
   * @return Rid with toplevel collection removed
   */
  public IRid removeRoot() {
    if (this.pathlen == 0) {
      return this;
    }

    int start = this.isAbsolute ? 1 : 0;
    int pos = this.path.indexOf(Rid.PATH_SEPARATOR, start);

    return (pos == -1 || pos == this.pathlen - 1) ? Rid.emptyRid : Rid.getRid(this.path.substring(pos), this.query);
  }

  /**
   * Remove the name of the resource, same as parent()
   *
   * @return parent of the resource
   */
  public IRid removeName() {
    if (this.pathlen == 0) {
      return this;
    }

    int pos;
    if (this.endsWithSlash) {
      pos = this.path.lastIndexOf(Rid.PATH_SEPARATOR, this.pathlen - 2);
    }
    else {
      pos = this.path.lastIndexOf(Rid.PATH_SEPARATOR);
    }

    switch (pos) {
      case 0:
        return Rid.rootRid;
      case -1:
        return Rid.emptyRid;
      default:
        return Rid.getRid(this.path.substring(0, pos));
    }
  }

  /**
   * Remove the (optional) extension of the resource name. Query parameters are
   * discarded.
   *
   * @return uri with extension of name removed
   */
  public IRid removeExtension() {
    if (this.pathlen == 0 || this.endsWithSlash) {
      return this;
    }

    int spos = this.path.lastIndexOf(Rid.PATH_SEPARATOR);
    int pos = this.path.lastIndexOf('.');

    if (pos > -1 && pos > spos) {
      return new Rid(this.path.substring(0, pos));
    }
    else {
      return this;
    }
  }

  /**
   * Remove a possible trailing slash from the Rid. Query parameters are
   * preserved.
   *
   * @return Rid without trailing slash
   */
  public IRid removeTrailingSlash() {
    return this.endsWithSlash ?
      Rid.getRid(this.path.substring(0, this.pathlen - 1), this.query) : this;
  }

  /**
   * Return if Rid is absolute, e.g. start with a slash.
   *
   * @return if Rid is absolute path
   */
  public boolean isAbsolute() {
    return this.isAbsolute;
  }

  /**
   * Return if Rid is root collection.
   *
   * @return if this is the root collection
   */
  public boolean isRoot() {
    return this.pathlen == 1 && this.isAbsolute;
  }

  /**
   * Determine if this Rid is an ancestor (parent or parent's parent, etc.) of
   * the given child Rid.
   *
   * @param child to test against
   * @return if this Rid is ancestor of child
   */
  public boolean isAncestorOf(IRid child) {
    if (this.pathlen < child.getPath().length() && this.isAbsolute == child.isAbsolute()) {
      if (child.getPath().startsWith(this.path)) {
        return this.endsWithSlash || child.getPath().charAt(this.pathlen) == Rid.PATH_SEPARATOR;
      }
    }
    return false;
  }

  /**
   * Determine if this Rid is an ancestor (parent or parent's parent, etc.) of
   * the given child Rid <em>or</em> the same.
   *
   * @param child to test against
   * @return if this Rid is ancestor of child
   */
  public boolean isAncestorOfOrSelf(IRid child) {
    if (this.equals(child)) {
      return true;
    }
    else if (this.pathlen < child.getPath().length() && this.isAbsolute == child.isAbsolute()) {
      if (child.getPath().startsWith(this.path)) {
        return this.endsWithSlash || child.getPath().charAt(this.pathlen) == Rid.PATH_SEPARATOR;
      }
    }
    return false;
  }

  /**
   * Return if this Rid ends with a slash.
   *
   * @return if this Rid ends with a slash
   */
  public boolean endsWithSlash() {
    return this.endsWithSlash;
  }

  /**
   * Concatenate this Rid with the given Rid. Treats this Rid as if it ends with
   * a slash and the parameter Rid as if it starts without a slash. <p>
   *
   * Note: do not use this method to add path segments, as the parameter is
   * parsed as legal Rid, not as a legal Rid segment; use {@link
   * Rid#addPathSegment(String)} instead.
   *
   * @param uri to append to this
   * @return new Rid as concatenation
   */
  public IRid add(String uri) {
    if (this.pathlen == 0) {
      return Rid.getRid(uri);
    }
    int ulen = uri.length();
    if (ulen == 0) {
      return this;
    }

    String query = null;
    int index = uri.lastIndexOf('?');
    if (index >= 0) {
      query = (index < ulen) ? uri.substring(index + 1) : "";
      uri = uri.substring(0, index);
      ulen = index;
    }

    StringBuffer sb = new StringBuffer(this.pathlen + ulen + 1);
    sb.append(this.path);

    if (this.endsWithSlash) {
      if (uri.charAt(0) == Rid.PATH_SEPARATOR) {
        sb.setLength(this.pathlen - 1);
      }
    }
    else {
      if (ulen == 0 || (uri.charAt(0) != Rid.PATH_SEPARATOR)) {
        sb.append(Rid.PATH_SEPARATOR);
      }
    }
    sb.append(uri);

    return Rid.getRid(sb.toString(), query);
  }

  /**
   * Concatenate this Rid with the given Rid. Treats this Rid as if it ends with
   * a slash and the parameter Rid as if it starts without a slash.
   *
   * @param uri to append to this
   * @return new Rid as concatenation
   */
  public IRid add(IRid uri) {
    if (this.pathlen == 0) {
      return uri;
    }
    if (uri.getPath().length() == 0) {
      return Rid.getRid(this.path, uri.getQuery());
    }

    StringBuffer sb = new StringBuffer(this.pathlen + uri.getPath().length() + 1);
    sb.append(this.path);

    if (this.endsWithSlash) {
      if (uri.isAbsolute()) {
        sb.setLength(this.pathlen - 1);
      }
    }
    else {
      if (!uri.isAbsolute()) {
        sb.append(Rid.PATH_SEPARATOR);
      }
    }
    sb.append(uri.getPath());

    return Rid.getRid(sb.toString(), uri.getQuery());
  }

  /**
   * Splits the Rid into its path components
   *
   * @return List of {@link String}s
   */
  public List split() {
    if (this.pathlen == 0) {
      return Collections.EMPTY_LIST;
    }
    return splitString(this.path, this.pathlen, Rid.PATH_SEPARATOR);
  }

  /**
   * Returns string representation of this Rid
   *
   * @return this Rid as string
   */
  public String toString() {
    if (this.uri == null) {
      if (this.query != null) {
        this.uri = this.path + '?' + this.query;
      }
      else {
        this.uri = this.path;
      }
    }
    return this.uri;
  }

  /**
   * Determine if this Rid equals other.
   *
   * @param other Rid
   * @return if this Rid equals other
   */
  public boolean equals(IRid other) {
    if (this == other) {
      return true;
    }
    if (other == null) {
      return false;
    }

    if (hashCode() != other.hashCode()) {
      return false;
    }

    if (this.pathlen == other.getPath().length()) {
      return this.path.equals(other.getPath()) && equalsQuery(other);
    }
    else if (this.endsWithSlash) {
      if (!other.endsWithSlash() && this.pathlen == other.getPath().length() + 1) {
        return this.path.startsWith(other.getPath()) && equalsQuery(other);
      }
    }
    else if (other.endsWithSlash()) {
      if (other.getPath().length() == this.pathlen + 1) {
        return other.getPath().startsWith(this.path) && equalsQuery(other);
      }
    }
    return false;
  }

  /**
   * Determine if this Rid equals other. Equality is defined for instances of
   * Rid only.
   *
   * @param other
   * @return if this Rid equals other
   */
  public boolean equals(Object other) {
    if (this == other) {
      return true;
    }

    if (other instanceof IRid) {
      return equals((IRid)other);
    }
    else {
      return false;
    }
  }

  public int hashCode() {
    if (this.hashCode == 0) {
      this.hashCode = (this.pathlen > 1 && this.endsWithSlash) ?
        this.path.substring(0, this.pathlen - 1).hashCode() : this.path.hashCode();
      if (this.query != null) {
        this.hashCode += this.query.hashCode();
      }
    }
    return this.hashCode;
  }

  /**
   * Return length of Rid in string representation
   *
   * @return length of Rid in string representation
   */
  public int length() {
    if (this.query != null) {
      return this.pathlen + 1 + this.query.length();
    }
    return this.pathlen;
  }

  private final static String RID_PATH_SEPARATOR = String.valueOf(Rid.PATH_SEPARATOR);

  /**
   * Resolve relative Rids
   *
   * @param uri An Rid, which may be relative or absolute
   * @return TBD: Description of the outgoing return value
   */
  public IRid resolveRelativeRid(String uri) {

    // absolute href (e.g. http:// c:\ c:/).
    if (this.pathlen > 0 && !(uri.startsWith(Rid.RID_PATH_SEPARATOR)) && uri.indexOf(":/") < 0 && uri.indexOf(":\\", 1) < 0) {

      // relative uri. Append it to the base url
      StringBuffer sb = new StringBuffer(128);
      if (this.endsWithSlash) {
        sb.append(this.path);
      }
      else {
        sb.append(parent().toString());
        if (sb.charAt(sb.length() - 1) != Rid.PATH_SEPARATOR) {
          sb.append(Rid.PATH_SEPARATOR);
        }
      }

      Rid tmp = Rid.getRid(uri);
      String query = tmp.getQuery();
      sb.append(tmp.getPath());

      sb = HttpUrl.normalizePath(sb);

      return Rid.getRid(sb.toString(), query);
//      //  remove all "./" where "." is a complete path segment
//      int index = -1;
//      while ((index = newUri.indexOf("/./")) != -1) {
//        newUri = path.substring(0, index+1).concat(newUri.substring(index+3));
//      }
//
//      // remove all "<segment>/../" where "<segment>" is a complete
//      // path segment not equal to ".."
//      index = -1;
//      int segIndex = -1;
//      String tempString = null;
//
//      while ((index = newUri.indexOf("/../")) > 0) {
//        tempString = newUri.substring(0, newUri.indexOf("/../"));
//        segIndex = tempString.lastIndexOf('/');
//        if (segIndex != -1) {
//          if (!tempString.substring(segIndex++).equals("..")) {
//            newUri = newUri.substring(0, segIndex).concat(newUri.substring(index+4));
//          }
//        }
//      }
//      return new Rid(newUri);
    }
    else {
      return Rid.getRid(uri);
    }
  }

  /**
   * Adds a path segment.
   *
   * @param segment String containing the new path segment
   * @return new Rid
   * @throws WcmException if segment contains characters that aren't allowed in
   *      path segments
   */
  public IRid addPathSegment(String segment)
    throws WcmException {
    if (segment.indexOf(Rid.PATH_SEPARATOR) >= 0) {
      throw new WcmException("illegal character for Rid path segment");
    }

    StringBuffer sb = new StringBuffer(this.pathlen + segment.length() + 1);
    sb.append(this.path);

    if (!this.endsWithSlash) {
      sb.append(Rid.PATH_SEPARATOR);
    }
    sb.append(segment);

    return Rid.getRid(sb.toString(), null);
  }

  /**
   * Create a new Rid by setting the given parameters as query part of this Rid.
   *
   * @param parameter to set to query
   * @return new Rid with parameter in query
   * @deprecated as of NW04. Use setQueryParameter
   */
  public IRid addQueryParameter(Properties parameter) {
    return setQueryParameter(parameter);
  }

  /**
   * Create a new Rid by setting the given parameters as query part of this Rid.
   *
   * @param parameter to set to query
   * @return new Rid with parameter in query
   */
  public IRid setQueryParameter(Properties parameter) {
    return Rid.getRid(this.path, UriCodec.EncodeQuery(parameter));
  }

  // --- Comparable interface ---

  public int compareTo(Object o) {
    String uri = this.path;
    String s2 = null;

    if (o instanceof String) {
      s2 = (String)o;
    }
    else if (o instanceof Rid) {
      s2 = ((Rid)o).path;
    }
    else {
      throw new java.lang.IllegalArgumentException("Rid cannot be compared to " + o.getClass().getName());
    }

    final int o_len = s2.length();
    if (this.pathlen != o_len) {
      if ((this.pathlen > 1) && (o_len > 1)) {
        if (this.endsWithSlash) {
          uri = uri.substring(0, this.pathlen - 1);
        }
        if (s2.charAt(o_len - 1) == Rid.PATH_SEPARATOR) {
          s2 = s2.substring(0, o_len - 1);
        }
      }
    }

    return uri.compareTo(s2);
  }

  // ----------------------------- private -----------------------------------

  private static List splitString(final String str, final int len, final int delim) {

    if (len == 0) {
      return Collections.EMPTY_LIST;
    }

    List names = new ArrayList(10);

    int i = (str.charAt(0) == delim) ? 1 : 0;
    int last = i;
    for (; i < len; ++i) {
      if (str.charAt(i) == delim) {
        names.add(str.substring(last, i));
        last = i + 1;
      }
    }

    if (last < i) {
      names.add(str.substring(last));
    }

    return names;
  }

  private Rid() {
    this(null);
  }

  private boolean equalsQuery(IRid other) {
    if (this.query != null) {
      return this.query.equals(other.getQuery());
    }
    else {
      return other.getQuery() == null;
    }
  }
}
