/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common;

import java.util.*;

/**
 * A Rid is an identifier for a CM resource. It holds all information to
 * identify a CM resource and retrieve it via the CM Framework. Each CM resource
 * has its unique Rid. The first pathsegment must be a valid name of a
 * repository (prefix). <p>
 *
 * A Rid has a path-like structure, like a path in a file system and follows the
 * following production: <pre>
 *   Rid         ::= '/' pathsegment | '/' pathsegment '?' query
 *   pathsegment ::= name | name '/' pathsegment | empty
 *   name        ::= any char except '/'
 *   query       ::= parameter | parameter '&' query | empty
 *   parameter   ::= varname | varname '=' value
 *   varname         ::= any char except '=', '&'   value       ::= any
 * character except '&'
 * </pre> Equality is defined as ignoring trailing '/' characters. Thus, the
 * Rids "/docs/" and /docs" are considered equal. There is special handling for
 * the root collection "/", which is not equal to the empty Rid "". <p>
 *
 * <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author frank.renkes@sapportals.com
 * @author stefan.eissing@greenbytes.de
 * @author m.breitenfelder@sapportals.com
 * @created 31. Januar 2003
 */
public interface IRid {
  /**
   * Convert this Rid to a representation usable in RFC2396 URI references.
   * Non-ASCII and reserved characters are encoded.<p>
   *
   * Note that such an encoded string cannot be used to construct an new Rid
   * object! The path in Rid objects is never, ever encoded.
   *
   * @return Rid suitable escaped as RFC2396 URI reference
   */
  public String toExternalForm();

  /**
   * Return the path component of this Rid.
   *
   * @return the path component
   */
  public String getPath();

  /**
   * Return the query component of this Rid (null if it does not exist).
   *
   * @return the query component
   */
  public String getQuery();

  /**
   * Get the query parameters of the Rid. Will return an empty Properties object
   * if Rid has no query part. Modifying the {@link Properties} object has no
   * effect on the Rid itself.
   *
   * @return query parameters as {@link Properties}
   */
  public Properties getQueryParameter();

  /**
   * Get Rid of toplevel collection (below root) of this Rid. <p>
   *
   * The root Rid of a toplevel collection is the root collection Rid. The root
   * Rid of the root Rid is the root Rid itself.
   *
   * @return Rid of toplevel collection below root
   */
  public IRid root();

  /**
   * Get the name of the resource, designated by this Rid. If the Rid has query
   * parameter, these are discarded.
   *
   * @return the name of the resource, designated by this Rid
   */
  public IRid name();


  /**
   * Get the Rid of the parent collection of this Rid
   *
   * @return the Rid of the parent collection
   */
  public IRid parent();

  /**
   * Get the extension part of the resource name. Returns the empty string if
   * there is no extension.
   *
   * @return the extension part of the resource name
   */
  public String extension();

  /**
   * Get a new Rid with the toplevel collection removed
   *
   * @return Rid with toplevel collection removed
   */
  public IRid removeRoot();

  /**
   * Remove the name of the resource, same as parent()
   *
   * @return parent of the resource
   */
  public IRid removeName();

  /**
   * Remove the (optional) extension of the resource name. Query parameters are
   * discarded.
   *
   * @return uri with extension of name removed
   */
  public IRid removeExtension();

  /**
   * Remove a possible trailing slash from the Rid. Query parameters are
   * preserved.
   *
   * @return Rid without trailing slash
   */
  public IRid removeTrailingSlash();

  /**
   * Return if Rid is absolute, e.g. start with a slash.
   *
   * @return if Rid is absolute path
   */
  public boolean isAbsolute();

  /**
   * Return if Rid is root collection.
   *
   * @return if this is the root collection
   */
  public boolean isRoot();

  /**
   * Determine if this Rid is an ancestor (parent or parent's parent, etc.) of
   * the given child Rid.
   *
   * @param child to test against
   * @return if this Rid is ancestor of child
   */
  public boolean isAncestorOf(IRid child);

  /**
   * Determine if this Rid is an ancestor (parent or parent's parent, etc.) of
   * the given child Rid <em>or</em> the same.
   *
   * @param child to test against
   * @return if this Rid is ancestor of child
   */
  public boolean isAncestorOfOrSelf(IRid child);

  /**
   * Return if this Rid ends with a slash.
   *
   * @return if this Rid ends with a slash
   */
  public boolean endsWithSlash();

  /**
   * Concatenate this Rid with the given Rid. Treats this Rid as if it ends with
   * a slash and the parameter Rid as if it starts without a slash.
   *
   * @param uri to append to this
   * @return new Rid as concatenation
   */
  public IRid add(String uri);

  /**
   * Concatenate this Rid with the given Rid. Treats this Rid as if it ends with
   * a slash and the parameter Rid as if it starts without a slash.
   *
   * @param uri to append to this
   * @return new Rid as concatenation
   */
  public IRid add(IRid uri);

  /**
   * Splits the Rid into its path components
   *
   * @return List of {@link String}s
   */
  public List split();

  /**
   * Return length of Rid in string representation
   *
   * @return length of Rid in string representation
   */
  public int length();

  /**
   * Resolve relative Rids
   *
   * @param uri An Rid, which may be relative or absolute
   * @return resolved Rid
   */
  public IRid resolveRelativeRid(String uri);

  /**
   * Adds a path segment.
   *
   * @param segment String containing the new path segment
   * @return new Rid
   * @exception Exception Exception raised in failure situation
   */
  public IRid addPathSegment(String segment)
    throws Exception;

  /**
   * Create a new Rid by setting the given parameters as query part of this Rid.
   *
   * @param parameter to set to query
   * @return new Rid with parameter in query
   */
  public IRid setQueryParameter(Properties parameter);
}
