/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common;

/**
 * A <a href="http://www.ietf.org/rfc/rfc2396.txt">RFC2396</a> URI interface.
 * <pre>
 *    remainder   ::= '//' authority abspath query
 *    abspath     ::= '/' pathsegment | empty
 *    pathsegment ::= name | name '/' pathsegment | empty
 *    query       ::= '?' uri-characters | empty
 * </pre> <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author julian.reschke@greenbytes.de
 * @author stefan.eissing@greenbytes.de
 * @created 20. Mrz 2003
 * @version $Id: //javabas/com.sapportals.wcm/50_COR/src/java/nemesis/common/com/sapportals/wcm/repository/common/IHierarchicalUri.java#4
 *      $
 */

public interface IHierarchicalUri extends IAbsoluteUri {

  /**
   * Append the given path segment to the current path of this uri. Takes care
   * of leading, trailing slashes. Note that the path <b>must</b> be uri encoded
   * or the resulting uri is not valid.
   *
   * @param path TBD: Description of the incoming method parameter
   * @return new uri with concatenated path
   */
  public IHierarchicalUri appendPath(String path);

  /**
   * Return authority part of this hierarchical Uri.
   *
   * @return authority part of this hierarchical Uri
   */
  public String getAuthority();

  /**
   * Return path of this Uri or, if not there, the emtpy string
   *
   * @return path of this uri
   */
  public String getPath();

  /**
   * Return the segments of the uri path
   *
   * @return segments of uri path
   */
  public String[] getPathSegments();

  /**
   * Return query of this Uri in RFC 2396 encoded form or, if not there, null
   *
   * @return query of this uri in RFC 2396 encoded form or null
   */
  public String getQuery();

  /**
   * Returns the uri of the root collection (path = "/").
   *
   * @return uri of server root
   */
  public IHierarchicalUri getRoot();

  /**
   * Determine if this Uri is ancestor of other uri
   *
   * @param other TBD: Description of the incoming method parameter
   * @return if this Uri is ancestor of other uri
   */
  public boolean isAncestorOf(IHierarchicalUri other);

  /**
   * Convert the given uri to a relative uri reference with this uri as base. If
   * this uri is no ancestor of the other uri, an absolute uri refernce is
   * returned.
   *
   * @param other uri to unresolve
   * @param fragment to add to result uri reference
   * @return uri reference relative to this uri
   */
  public IUriReference unresolve(IHierarchicalUri other, String fragment);

  /**
   * TBD: Description of the Method.
   *
   * @param ref TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public Rid mapToResourceID(IUriReference ref);

  /**
   * TBD: Description of the Method.
   *
   * @param ref TBD: Description of the incoming method parameter
   * @param codec TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public Rid mapToResourceID(IUriReference ref, UriCodec codec);

  /**
   * TBD: Description of the Method.
   *
   * @param wcmPath TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public IUriReference mapToAbsolutePath(Rid wcmPath);

  /**
   * TBD: Description of the Method.
   *
   * @param wcmPath TBD: Description of the incoming method parameter
   * @param codec TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public IUriReference mapToAbsolutePath(Rid wcmPath, UriCodec codec);

  /**
   * Given this Uri as the base, map the uri reference to a uri reference with
   * absolute path component. <br>
   * <pre>
   * Example: "http://host/docs" + "test%203", gives "/docs/test%203"
   * Example: "http://host/docs" + "http://another/fs/test", gives "http://another/fs/test"
   * </pre>
   *
   * @param ref TBD: Description of the incoming method parameter
   * @return the mapped uri reference
   */
  public IUriReference mapToAbsolutePath(IUriReference ref);

  /**
   * TBD: Description of the Method.
   *
   * @param wcmPath TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public IHierarchicalUri mapToAbsoluteUri(Rid wcmPath);

  /**
   * TBD: Description of the Method.
   *
   * @param wcmPath TBD: Description of the incoming method parameter
   * @param codec TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public IHierarchicalUri mapToAbsoluteUri(Rid wcmPath, UriCodec codec);

  /**
   * Construct a new object of same type with the given query. If query is
   * <code>null</code> , the new uri will be without query. The query string
   * must be in RFC 2396 encoded form. See {@link UriQuery} for manipulation of
   * query strings.
   *
   * @param query encoded query string for returned object
   * @return hierarchical uri of same type as this
   */
  public IHierarchicalUri setQuery(String query);

  /**
   * Construct a new object of same type with the given path. If path is <code>
   * null</code> , the new uri will have the root path /. The path string must
   * be in RFC 2396 encoded form. See {@link URICodec}. The query of the
   * resulting uri will always be empty.
   *
   * @param path encoded path string for returned object
   * @return hierarchical uri of same type as this
   */
  public IHierarchicalUri setPath(String path);
}
