/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common;

import java.util.StringTokenizer;

/**
 * An {@link IHierarchicalUri} implementation for HTTP/HTTPS URLs. <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author julian.reschke@greenbytes.de
 * @author stefan.eissing@greenbytes.de
 * @version $Id: HttpUrl.java,v 1.2 2003/04/28 10:08:34 jre Exp $
 */

public class HttpUrl implements IHierarchicalUri {

  private final static String SCHEME_HTTP = "http";
  private final static String SCHEME_HTTPS = "https";

  private final static int HTTP_DEFAULT_PORT = 80;
  private final static int HTTPS_DEFAULT_PORT = 443;

  private final String m_scheme;
  private final String m_host;
  private final int m_port;
  private final String m_authority;
  private final String m_path;
  private final String m_query;

  private final String m_full;

  private String[] m_pathSegments;

  private int m_hashCode;

  public static int DEFAULT_PORT = -1;

  public HttpUrl(String scheme, String host, int port, String path, String query)
    throws IllegalArgumentException {
    scheme = scheme.toLowerCase();

    if (scheme.equals(SCHEME_HTTP)) {
      scheme = SCHEME_HTTP;
      if (port == DEFAULT_PORT) {
        port = HTTP_DEFAULT_PORT;
      }
    }
    else if (scheme.equals(SCHEME_HTTPS)) {
      scheme = SCHEME_HTTPS;
      if (port == DEFAULT_PORT) {
        port = HTTPS_DEFAULT_PORT;
      }
    }
    else {
      throw new IllegalArgumentException("scheme must be http: or https:");
    }

    m_scheme = scheme;
    m_host = host.toLowerCase();
    m_port = port;
    m_path = (path != null) ? path : "";
    m_query = query;

    StringBuffer sb = new StringBuffer(128);
    sb.append(m_host);
    if ((m_scheme == SCHEME_HTTP && m_port != HTTP_DEFAULT_PORT)
       || (m_scheme == SCHEME_HTTPS && m_port != HTTPS_DEFAULT_PORT)) {
      sb.append(':').append(m_port);
    }
    m_authority = sb.toString();

    sb.setLength(0);
    sb.append(m_scheme).append("://").append(m_authority).append(m_path);
    if (m_query != null) {
      sb.append('?').append(m_query);
    }
    m_full = sb.toString();
  }

  public HttpUrl(String scheme, String authority, String path, String query)
    throws IllegalArgumentException {
    scheme = scheme.toLowerCase();

    int defport;
    if (scheme.equals(SCHEME_HTTP)) {
      scheme = SCHEME_HTTP;
      defport = HTTP_DEFAULT_PORT;
    }
    else if (scheme.equals(SCHEME_HTTPS)) {
      scheme = SCHEME_HTTPS;
      defport = HTTPS_DEFAULT_PORT;
    }
    else {
      throw new IllegalArgumentException("scheme must be http: or https:");
    }

    m_scheme = scheme;
    authority = authority.toLowerCase();
    m_path = (path != null) ? path : "";
    m_query = query;

    int port = defport;
    String host = authority;
    int pos = authority.lastIndexOf(':');

    if (pos > 0) {
      // Beware: RFC 2732 defines IPV6 addresses for URIs. We only found
      // a valid port number definition, if
      // - the authority did not start with a '['
      // or
      // - it did start with '[' and the character before the last ':' is a ']'
      //   (http://[ipv6address]:port/)
      //
      if (authority.charAt(0) != '[' || authority.charAt(pos - 1) == ']') {
        try {
          String portString = authority.substring(pos + 1);
          if (portString.length() > 0) {
            port = Integer.parseInt(portString);
          }
          host = authority.substring(0, pos);
        }
        catch (NumberFormatException ex) {
          //$JL-EXC$ 
          throw new IllegalArgumentException(ex.getMessage());
        }
      }
    }

    m_host = host;
    m_port = port;

    if ((m_scheme == SCHEME_HTTP && m_port == HTTP_DEFAULT_PORT)
       || (m_scheme == SCHEME_HTTPS && m_port == HTTPS_DEFAULT_PORT)) {
      authority = m_host;
    }

    m_authority = authority;

    StringBuffer sb = new StringBuffer(128);
    sb.append(m_scheme).append("://").append(m_authority).append(m_path);
    if (m_query != null) {
      sb.append('?').append(m_query);
    }

    m_full = sb.toString();
  }

  public boolean equals(Object other) {
    if (other == null) {
      return false;
    }
    else if (other instanceof HttpUrl) {
      return equals((HttpUrl)other);
    }
    else if (other instanceof IHierarchicalUri) {
      return equals((IHierarchicalUri)other);
    }
    else if (other instanceof IAbsoluteUri) {
      return equals((IAbsoluteUri)other);
    }

    return false;
  }

  public boolean equals(IAbsoluteUri other) {
    if (other == null) {
      return false;
    }
    if (this == other) {
      return true;
    }
    if (other instanceof HttpUrl) {
      return equals((HttpUrl)other);
    }
    else if (other instanceof IHierarchicalUri) {
      return equals((IHierarchicalUri)other);
    }

    return false;
  }

  public boolean equals(IHierarchicalUri other) {
    if (other == null) {
      return false;
    }
    if (this == other) {
      return true;
    }
    if (other instanceof HttpUrl) {
      return equals((HttpUrl)other);
    }

    return m_scheme.equalsIgnoreCase(other.getScheme())
       && ((m_query != null && m_query.equals(other.getQuery()))
       || (m_query == null && other.getQuery() == null))
       && m_path.equals(other.getPath())
       && m_authority.equalsIgnoreCase(other.getAuthority());
  }

  public boolean equals(HttpUrl other) {
    if (other == null) {
      return false;
    }
    if (this == other) {
      return true;
    }
    return m_scheme == other.m_scheme
       && ((m_query != null && m_query.equals(other.m_query))
       || (m_query == null && other.m_query == null))
       && m_path.equals(other.m_path)
       && m_authority.equalsIgnoreCase(other.m_authority);
  }

  /**
   * Append the given path segment to the current path of this uri. Takes care
   * of leading, trailing slashes. Note that the path <b>must</b> be uri encoded
   * or the resulting uri is not valid.
   *
   * @param path TBD: Description of the incoming method parameter
   * @return new uri with concatenated path
   */
  public IHierarchicalUri appendPath(String path) {
    return new HttpUrl(
      m_scheme, m_host, m_port, normalizePath(concatPath(m_path, path, true)).toString(), m_query);
  }

  /**
   * Get the scheme of this Uri (without ':').
   *
   * @return scheme of this Uri
   */
  public String getScheme() {
    return m_scheme;
  }

  /**
   * Get the remainder of this Uri (part after scheme without ':').
   *
   * @return remainder of this Uri
   */
  public String getRemainder() {
    return m_full.substring(m_scheme.length() + 1);
  }

  /**
   * Return authority part of this hierarchical Uri.
   *
   * @return authority part of this hierarchical Uri
   */
  public String getAuthority() {
    return m_authority;
  }

  /**
   * Return the host name of this uri
   *
   * @return the host name
   */
  public String getHost() {
    return m_host;
  }

  /**
   * Return the port number of this uri
   *
   * @return the port number
   */
  public int getPort() {
    return m_port;
  }

  /**
   * Return path of this Uri or, if not there, the emtpy string
   *
   * @return path of this uri
   */
  public String getPath() {
    return m_path;
  }

  /**
   * Return the segments of the uri path
   *
   * @return segments of uri path
   */
  public String[] getPathSegments() {
    if (m_pathSegments == null) {
      synchronized (this) {
        if (m_pathSegments == null) {
          StringTokenizer tok = new StringTokenizer(m_path, "/");
          int tokens = tok.countTokens();
          m_pathSegments = new String[tokens];
          for (int i = 0; i < tokens; ++i) {
            m_pathSegments[i] = tok.nextToken();
          }
        }
      }
    }
    return m_pathSegments;
  }

  /**
   * Return query of this Uri or, if not there, null
   *
   * @return query of this uri or null
   */
  public String getQuery() {
    return m_query;
  }

  /**
   * Returns the uri of the root collection (path = "/").
   *
   * @return uri of server root
   */
  public IHierarchicalUri getRoot() {
    if (m_query == null && "/".equals(m_path)) {
      return this;
    }
    else {
      return new HttpUrl(m_scheme, m_host, m_port, "/", null);
    }
  }

  public String toString() {
    return m_full;
  }

  public int hashCode() {
    if (m_hashCode == 0) {
      m_hashCode = m_full.hashCode();
    }
    return m_hashCode;
  }

  /**
   * Determine if this Uri is ancestor of other uri
   *
   * @param other TBD: Description of the incoming method parameter
   * @return if this Uri is ancestor of other uri
   */
  public boolean isAncestorOf(IHierarchicalUri other) {
    if (m_scheme.equalsIgnoreCase(other.getScheme())
       && m_authority.equalsIgnoreCase(other.getAuthority())) {
      String opath = other.getPath();
      return (opath.startsWith(m_path)
         && (m_path.length() == opath.length()
         || (m_path.length() > 0 && m_path.charAt(m_path.length() - 1) == '/')
         || opath.charAt(m_path.length()) == '/'));
    }
    return false;
  }

  /**
   * Resolve the uri reference in the context of this Uri.
   *
   * @param reference TBD: Description of the incoming method parameter
   * @return resolved reference as absolute uri without fragment identifier
   */
  public IAbsoluteUri resolve(IUriReference reference) {
    if (reference.isAbsolute()) {
      return reference.getUri();
    }
    else {
      String rscheme = reference.getScheme();
      if (rscheme != null && !m_scheme.equalsIgnoreCase(rscheme)) {
        // degenerate case: see RFC 2396 page 32
        throw new java.lang.IllegalArgumentException(
          "reference " + reference + " cannot be resolved with " + this);
      }

      String rauthority = reference.getAuthority();
      String rpath = reference.getPath();
      String rquery = reference.getQuery();

      if (rauthority == null) {
        int rlen = rpath.length();
        if (rlen == 0 && rquery == null) {
          return this;
        }

        if (rlen == 0 || rpath.charAt(0) != '/') {
          StringBuffer sb = concatPath(m_path, rpath, false);
          rpath = normalizePath(sb).toString();
        }

        return new HttpUrl(m_scheme, m_host, m_port, rpath, rquery);
      }
      else {
        return new HttpUrl(m_scheme, rauthority, rpath, rquery);
      }
    }
  }

  /**
   * Convert the given uri to a relative uri reference with this uri as base. If
   * this uri is no ancestor of the other uri, an absolute uri refernce is
   * returned.
   *
   * @param other uri to unresolve
   * @param fragment to add to result reference
   * @return uri reference relative to this uri
   */
  public IUriReference unresolve(IHierarchicalUri other, String fragment) {
    if (!m_scheme.equalsIgnoreCase(other.getScheme())
       || !m_authority.equalsIgnoreCase(other.getAuthority())) {
      return new UriReference(other, fragment);
    }

    // same scheme and authority
    //
    String rpath;
    String opath = other.getPath();
    if (opath.startsWith(m_path)) {
      int index = m_path.lastIndexOf('/');
      if (index >= 0) {
        ++index;
        rpath = (index < opath.length()) ? opath.substring(index) : "";
      }
      else {
        rpath = opath;
      }
    }
    else {
      String[] segments = getPathSegments();
      String[] osegments = other.getPathSegments();
      int i;
      for (i = 0; i < segments.length && i < osegments.length; ++i) {
        if (!segments[i].equals(osegments[i])) {
          break;
        }
      }

      StringBuffer sb = new StringBuffer(128);
      int depth = segments.length - i;
      if (m_path.charAt(m_path.length() - 1) != '/') {
        --depth;
      }

      for (int j = 0; j < depth; ++j) {
        if (sb.length() == 0) {
          sb.append("..");
        }
        else {
          sb.append("/..");
        }
      }

      for (int j = i; j < osegments.length; ++j) {
        if (sb.length() != 0) {
          sb.append('/');
        }
        sb.append(osegments[j]);
      }
      if (osegments.length > 0 && opath.charAt(opath.length() - 1) == '/') {
        sb.append('/');
      }

      rpath = sb.toString();
    }

    return new UriReference(null, rpath, other.getQuery(), fragment);
  }

  public Rid mapToResourceID(IUriReference ref) {
    return mapToResourceID(ref, null);
  }


  public Rid mapToResourceID(IUriReference ref, UriCodec codec) {
    if (ref.isAbsolute()) {
      IAbsoluteUri uri = ref.getUri();
      if (!(uri instanceof IHierarchicalUri)) {
        return null;
      }
      IHierarchicalUri huri = (IHierarchicalUri)uri;
      if (!m_scheme.equals(huri.getScheme())
         || !m_authority.equalsIgnoreCase(huri.getAuthority())) {
        return null;
      }
    }

    String path = ref.getPath();
    if (path.length() > 0 && path.charAt(0) != '/') {

      StringBuffer sb = concatPath(m_path, path, false);
      path = normalizePath(sb).toString();
    }

    int mlen = m_path.length();
    if (mlen > 0) {
      if (!path.startsWith(m_path)) {
        return null;
      }

      if (m_path.charAt(mlen - 1) == '/') {
        --mlen;
      }
    }

    if (codec != null) {
      path = codec.decode(path.substring(mlen));
    }
    else {
      path = UriCodec.Decode(path.substring(mlen));
    }

    return new Rid(path, ref.getQuery());
  }


  /**
   * Given this Uri as the base, map the uri reference to a uri reference with
   * absolute path component. <br>
   * <pre>
   * Example: "http://host/docs" + "test%203", gives "/test%203"
   * Example: "http://host/docs/" + "test%203", gives "/docs/test%203"
   * Example: "http://host/docs" + "http://another/docs/test", gives "http://another/docs/test"
   * </pre>
   *
   * @param ref TBD: Description of the incoming method parameter
   * @return the mapped uri reference
   */
  public IUriReference mapToAbsolutePath(IUriReference ref) {
    if (ref.isAbsolute()) {
      IAbsoluteUri uri = ref.getUri();
      if (uri instanceof IHierarchicalUri) {
        IHierarchicalUri huri = (IHierarchicalUri)uri;
        if (huri.getScheme().equalsIgnoreCase(m_scheme)
           && huri.getAuthority().equalsIgnoreCase(m_authority)) {
          return new UriReference(null, huri.getPath(), huri.getQuery(), ref.getFragmentIdentifier());
        }
      }
    }
    else {
      String path = ref.getPath();
      if (!path.startsWith("/")) {
        StringBuffer sb = concatPath(m_path, path, false);
        path = normalizePath(sb).toString();

        return new UriReference(null, path, ref.getQuery(), ref.getFragmentIdentifier());
      }
    }

    return ref;
  }

  public IUriReference mapToAbsolutePath(Rid wcmPath) {
    return mapToAbsolutePath(wcmPath, null);
  }

  public IUriReference mapToAbsolutePath(Rid wcmPath, UriCodec codec) {
    String path;
    String query = wcmPath.getQuery();
    if (codec != null) {
      path = codec.encodePath(wcmPath.getPath());
    }
    else {
      path = UriCodec.EncodePath(wcmPath.getPath());
    }

    path = concatPath(m_path, path, true).toString();

    return new UriReference(null, path, query, null);
  }

  public IHierarchicalUri mapToAbsoluteUri(Rid wcmPath) {
    return mapToAbsoluteUri(wcmPath, null);
  }

  public IHierarchicalUri mapToAbsoluteUri(Rid wcmPath, UriCodec codec) {
    String query = wcmPath.getQuery();
    String path = wcmPath.getPath();

    if (path.length() > 0) {
      if (codec != null) {
        path = codec.encodePath(path);
      }
      else {
        path = UriCodec.EncodePath(path);
      }

      path = concatPath(m_path, path, true).toString();
    }
    else {
      path = m_path;
    }

    return new HttpUrl(m_scheme, m_host, m_port, path, query);
  }

  public IHierarchicalUri setQuery(String query) {
    if (m_query == query || (m_query != null && m_query.equals(query))) {
      return this;
    }
    return new HttpUrl(m_scheme, m_host, m_port, m_path, query);
  }

  public IHierarchicalUri setPath(String path) {
    return new HttpUrl(m_scheme, m_host, m_port, path, null);
  }

  public String toExternalForm() {
    return toString();
  }

  // ------------------------ private -----------------------------------------

  private static StringBuffer concatPath(String path, String segment, boolean append) {
    int plen = path.length();
    if (plen == 0) {
      return new StringBuffer(segment);
    }
    int slen = segment.length();

    StringBuffer sb = new StringBuffer(plen + slen + 1);
    sb.append(path);
    if (path.charAt(plen - 1) == '/') {
      if (slen > 0) {
        if (segment.charAt(0) == '/') {
          sb.append(segment.substring(1));
        }
        else {
          sb.append(segment);
        }
      }
    }
    else {
      if (append) {
        if (slen > 0 && segment.charAt(0) != '/') {
          sb.append('/');
        }
      }
      else {
        int index = path.lastIndexOf('/');
        if (index >= 0) {
          sb.delete(index + 1, sb.length());
        }
        else {
          sb.setLength(0);
        }
      }
      sb.append(segment);
    }

    return sb;
  }

  private final static int NORMAL = 0;
  private final static int SLASH = 1;
  private final static int CURRENT = 2;
  private final static int PARENT = 3;

  protected static StringBuffer normalizePath(StringBuffer sb) {
    int state = NORMAL;
    int i = 0;
    while (i < sb.length()) {
      char c = sb.charAt(i);
      switch (state) {
        case NORMAL:
          if (c == '/') {
            state = SLASH;
          }
          break;
        case SLASH:
          if (c == '.') {
            state = CURRENT;
          }
          else if (c == '/') {
            // We are at the end of "//", remove one "/"
            sb.deleteCharAt(i);
            --i;
          }
          else {
            state = NORMAL;
          }
          break;
        case CURRENT:
          if (c == '.') {
            state = PARENT;
          }
          else if (c == '/') {
            // we are at the end of "/./", remove it
            sb.delete(i - 2, i);
            i -= 2;
            state = SLASH;
          }
          else {
            state = NORMAL;
          }
          break;
        case PARENT:
          if (c == '/') {
            // We are at the end of "/../", remove it up to the previous "/x"
            // where 'x' is any character except '.'
            //
            int lastSlash = -1;
            boolean seenNonDot = false;
            forloop :
            for (int j = i - 4; j >= 0; --j) {
              switch (sb.charAt(j)) {
                case '/':
                  if (seenNonDot) {
                    lastSlash = j;
                  }
                  break forloop;
                case '.':
                  break;
                default:
                  seenNonDot = true;
                  break;
              }
            }
            if (lastSlash >= 0) {
              sb.delete(lastSlash, i);
              i = lastSlash;
            }
            else {
              // We cannot go back any further, leave the .. as it is
              // However if we are at the beginning of the path,
              // remove /.. (same behaviour as IE)
              if (i == 3) {
                sb.delete(0, i);
                i = 0;
              }
            }
            state = SLASH;
          }
          else {
            state = NORMAL;
          }
          break;
        default:
          break;
      }
      ++i;
    }

    // end of buffer reached
    switch (state) {
      case CURRENT:
        // buffer ends in "/.", remove "."
        sb.delete(i - 1, sb.length());
        break;
      case PARENT:
        // buffer ends in "/..", remove up to previous "/" or remove ".."
        int lastSlash = -1;
        for (int j = i - 4; j >= 0; --j) {
          if (sb.charAt(j) == '/') {
            lastSlash = j;
            break;
          }
        }
        if (lastSlash >= 0) {
          sb.delete(lastSlash + 1, sb.length());
        }
        else {
          sb.delete(i - 2, sb.length());
        }
        break;
      default:
        break;
    }

    return sb;
  }

}
