/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.common;

/**
 * A <a href="http://www.ietf.org/rfc/rfc2396.txt">RFC2396</a> IUri
 * implementation. <pre>
 *     Uri ::= scheme ':' opaque_part
 * </pre> <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author julian.reschke@greenbytes.de
 * @author stefan.eissing@greenbytes.de
 * @version $Id: GenericUri.java,v 1.2 2003/04/28 10:08:34 jre Exp $
 */

public class GenericUri implements IAbsoluteUri {

  private final String scheme;
  private final String remainder;
  private String full;
  private int hashCode;

  public GenericUri(String scheme, String remainder)
    throws IllegalArgumentException {

    if (scheme == null) {
      throw new IllegalArgumentException("scheme must be non-null");
    }

    this.scheme = scheme.toLowerCase();
    this.remainder = (remainder != null) ? remainder : "";
  }

  /**
   * Get the scheme of this Uri (without ':').
   *
   * @return scheme of this Uri
   */
  public String getScheme() {
    return this.scheme;
  }

  /**
   * Get the remainder of this Uri (opaque part after scheme without ':').
   *
   * @return remainder of this Uri
   */
  public String getRemainder() {
    return this.remainder;
  }

  /**
   * Resolve the uri reference in the context of this Uri.
   *
   * @param ref reference to resolve
   * @return resolved reference as absolute uri without fragment identifier
   * @exception IllegalArgumentException Exception raised in failure situation
   */
  public IAbsoluteUri resolve(IUriReference ref)
    throws IllegalArgumentException {
    if (ref.isAbsolute()) {
      return ref.getUri();
    }
    else {
      String rscheme = ref.getScheme();
      if (rscheme != null && !this.scheme.equalsIgnoreCase(rscheme)) {
        throw new IllegalArgumentException("scheme does not match");
      }
      String rauth = ref.getAuthority();
      if (rauth != null) {
        throw new IllegalArgumentException("reference has authority");
      }
    }

    String rquery = ref.getQuery();
    return new GenericUri(this.scheme, ref.getPath() + (rquery != null ? rquery : ""));
  }

  public boolean equals(Object other) {
    if (other instanceof IAbsoluteUri) {
      return this.equals((IAbsoluteUri)other);
    }
    return false;
  }

  public boolean equals(IAbsoluteUri other) {
    if (other == null) {
      return false;
    }
    if (other == this) {
      return true;
    }
    return this.scheme.equalsIgnoreCase(other.getScheme())
       && this.remainder.equals(other.getRemainder());
  }

  public int hashCode() {
    if (this.hashCode == 0) {
      this.hashCode = this.toString().hashCode();
    }
    return this.hashCode;
  }

  public String toString() {
    if (this.full == null) {
      this.full = this.scheme + ":" + this.remainder;
    }
    return this.full;
  }

  public String toExternalForm() {
    return toString();
  }

  /**
   * Construct a new IUri with the scheme of this and the given remainder. used
   * when generic uri resolves references. This method should be overridden by
   * subclasses to construct their own class.
   *
   * @param remainder new opaque part of uri
   * @return new IUri instance
   * @exception IllegalArgumentException Exception raised in failure situation
   * @throws java.lang.ArgumentNotValidException when remainder is not legal
   */
  protected IAbsoluteUri newInstance(String remainder)
    throws IllegalArgumentException {
    return new GenericUri(this.scheme, remainder);
  }
}
