package com.sap.caf.rt.ui.cool.generic;

import java.lang.reflect.Constructor;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;

import javax.naming.Context;

import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.tc.col.cds.ConnectionData;
import com.sap.tc.col.client.generic.api.IMessageList;
import com.sap.tc.col.client.generic.api.IServiceFacade;
import com.sap.tc.col.client.generic.api.IServiceModule;
import com.sap.tc.col.client.metadata.api.IServiceModuleDescriptor;
import com.sap.tc.col.servicemanager.api.ISrvMgrCallbackOnFlush;
import com.sap.tc.col.servicemanager.api.ISrvMgrMessage;
import com.sap.tc.col.servicemanager.api.ISrvMgrServiceManager;
import com.sap.tc.col.servicemanager.api.calls.ISrvMgrCall;
import com.sap.tc.logging.Location;


/**
 * class <code>ServiceFacade</code> is the central entry point for the GCP User (Generic Client Proxy).
 * Beside the entry point function, there are the following main tasks:
 * <ul>
 * <li> factory for the <code>ServiceModule</code> instances
 * <li> delivers meta data information
 * <li> holds association to <code>ServiceManager</code>
 * </ul>
 * @author Helmut Mueller
 */
public class ServiceFacade implements IServiceFacade, ISrvMgrCallbackOnFlush {

	/** Logging properites for this class */
	private static final String APPLICATION	= ServiceFacade.class.getName();
	private static final String jARMRequest = AbstractModelClass.jARMReqPrefix+APPLICATION;
	private static final Location logger = Location.getLocation(APPLICATION);
  
  /**
   * constructor
   */
  public ServiceFacade(Locale locale) {
    init(null,locale);
  }
  
  /**
   * Constructor for test purpose 
   * @param url - connection url
   * @param user - user name
   * @param password - password
   */
  public ServiceFacade(String url, String user, String password) {
  	Hashtable prop = new Hashtable();
  	prop.put(Context.PROVIDER_URL, url);
	prop.put(Context.SECURITY_PRINCIPAL, user);
	prop.put(Context.SECURITY_CREDENTIALS, password);
	prop.put(Context.INITIAL_CONTEXT_FACTORY, "com.sap.engine.services.jndi.InitialContextFactoryImpl");
	init(prop, Locale.getDefault());
  }
  

  public ServiceFacade(Hashtable connection_properties, Locale locale) {
    init(connection_properties,locale);
  }

  
  /**
   * constructor 
   */
  public ServiceFacade(ConnectionData connectionData, Locale locale) {
    init(null, locale);
  }  
  /**
   * associated ServiceManager interface
   */
  private CAFServiceManager serviceManager;
  
  /**
   * returns the associated ServiceManager or an IllegalAccessException, if the
   * connection could not be established or is closed
   */
  protected ISrvMgrServiceManager getServiceManager() {
    if ( serviceManager == null )
      throw new IllegalStateException("ServiceManager is not initialized");
    return serviceManager;
  }
  
  /**
   * init connection and create ServiceManager, only one connection for a ServiceManager
   */
	protected void init(Hashtable connection_properties, Locale locale)
	{
		final String method = jARMRequest + ":init(Hashtable, Locale)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		try {
			//@TODO handle Locale
			//if ( locale != null ) 
			serviceManager = new CAFServiceManager(connection_properties,locale);
			//else
			//  serviceManager = ServiceManagerFactory.createServiceManager(client, username, password, language, server, system, debug);
		    
			// register this ServiceFacade as callback for flush
			serviceManager.registerCallbackOnFlush(this);
		} catch(Exception e) {
			logger.catching("service manager initialization failed", e);
			throw new GCPRuntimeException("service manager initialization failed", e);
	    } finally {  
    		CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		}
	}
  
   
  
  public String[] getServiceModuleNames() {
    return getServiceManager().getServiceModuleNames();
  }

  /**
   * returns the COL ServiceModule with given name
   * @param serviceModuleName the name of the service module
   * @return IServiceModule the service module
   */
  public IServiceModule getServiceModule(String serviceModuleName) {
    return getServiceModule(serviceModuleName,"");
  }
  
  /**
   * returns the meta data of the service module with given name.
   * @param serviceModuleName
   * @return ServiceModuleDescriptor
   */
  public IServiceModuleDescriptor getServiceModuleDescriptor(String serviceModuleName) {
    return ((CAFServiceManager)getServiceManager()).getServiceModuleDescriptor(serviceModuleName);
  }
  
  /**
   * associated ServiceModules
   */
  private HashMap serviceModules = new HashMap();
  
  /**
   * flushes the service queue, which is associated with this <code<ServiceManager</code> instance.
   * the resulting events beforeFlush, and afterFlush are sent via callback mechanism.
   */
  public void flush() {
    // call ServiceManager flush
    getServiceManager().flush("", "");
  }
  
  /**
   * returns <code>true</code> if changing calls (insert, update, delete, Action) are pending or not commited, or
   * <code>false</code> otherwise. The method flushes the queue implicitly and then returns the result.
   * @return boolean <code>true</code> if changing calls (insert, update, delete, Action) are pending or not commited, or
   * <code>false</code> otherwise
   */
  public boolean isDirty() {
	for (Iterator it = serviceModules.values().iterator(); it.hasNext();) {
		// just check service modules dirty state
		if (((ServiceModule) it.next()).isDirty()) {
			return true;
		}
	}
	return false;
  }

  /**
   * global message list for all messages in all service modules belonging 
   * to this service facade
   */
  private final MessageList allMessages = new MessageList();

  /**
   * Message list for out-of-band messages (not associated to a SvrMgrCall). 
   * This is a child list of allMessages.
   */
  private final MessageList outOfBandMessages = new MessageList(allMessages);

  public IMessageList getAllMessages() {
	allMessages.copyMessagesFrom((MessageList) getAndClearSystemMessages());
    return allMessages;
  }  

  MessageList getAllMessagesInternal() {
    return allMessages;
  }  

  public IMessageList getMessages() {
    return outOfBandMessages;
  }  

  /**
   * adds outbound messages from service manager to message list
   * called after flush
   */
  private void addMessages() {
    ISrvMgrMessage[] srvMgrMessages = getServiceManager().getMessages();
    if ( srvMgrMessages != null ) {
      for(int i=0; i<srvMgrMessages.length; i++) {
        Message message = new Message(srvMgrMessages[i]);
        outOfBandMessages.addMessage(message);
      }
    }
  }

  /**
   * Performs a rollback on the current backend transaction.
   * A new transaction is started implicitly.
   * 
   */
  public void cleanup() {
    getServiceManager().cleanup();
  }

  /**
   * Performs a commit on the current backend transaction.
   * A new transaction is started implicitly.
   * 
   * @return true if the save operation succeeded, false if it was rejected,
   */
  public boolean save() {
    return getServiceManager().save();
  }
  
  /**
   * init logging in standalone modus
   */
  public void initLocalLogging() {
// !WARNING commented out because all logging should be set up from Admin Tool
//    Properties loggingProperties = new Properties();
//    try {
//      loggingProperties.load(ServiceFacade.class.getResourceAsStream("logging.properties"));
//      new PropertiesConfigurator(loggingProperties).configure();
//    } catch (Exception ex) {
//	      logger.warningT("ServiceFacade: Could not load default logging configuration for GCP.");
//	      logger.catching(ex);
//    }
  }
  
  /**
   * set backend debugging mode
   */  
  public void setDebug(boolean enabled, String serviceModuleName, String guiHost)
  {
    getServiceManager().setDebug(enabled, serviceModuleName, guiHost);
  }
  
  
  public void defineServiceModuleDefaultImpl(Class clazz) {
    if ( clazz == null )
      throw new GCPRuntimeException("default implementation class must not be null");
    defineServiceModuleImpl0(null, clazz);
  }

  public void defineServiceModuleImpl(String serviceModuleName, Class clazz) {
    if ( serviceModuleName == null || serviceModuleName.length() == 0 )
      throw new IllegalArgumentException("service modul name must not be null");
    if ( clazz == null )
      throw new IllegalArgumentException("implementation class must not be null");
    defineServiceModuleImpl0(serviceModuleName, clazz);
  }

  private Map serviceModuleImplementations = new HashMap();

  {
    defineServiceModuleDefaultImpl(ServiceModule.class);
  }
  
   
  private static Class[] SERVICE_MODULE_CTOR_SIGNATURE = { IServiceFacade.class,CAFServiceManager.class, 
												IServiceModuleDescriptor.class };

  private void defineServiceModuleImpl0(String serviceModuleName, Class clazz) {
		final String method = jARMRequest + ":defineServiceModuleImpl0(String, Class)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
    try {
			if ( !ServiceModule.class.isAssignableFrom(clazz) )
			  throw new IllegalArgumentException("class " + clazz.getName() + " doesn't fulfill the contract for ServiceModule implementations");
      Constructor ctor = clazz.getDeclaredConstructor(SERVICE_MODULE_CTOR_SIGNATURE);    
      serviceModuleImplementations.put(serviceModuleName, ctor);
      logger.infoT("service module implementation " + serviceModuleName + " registered");
    }
    catch (NoSuchMethodException e) {
      logger.catching(e);
      throw new IllegalArgumentException("class " + clazz.getName() + " doesn't fulfill the contract for ServiceModule implementations");
    }
    finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
    }
  }
  
  

  
  /**
   * @see com.sap.tc.col.servicemanager.api.ISrvMgrCallbackOnFlush#afterFlush(ISrvMgrCall[])
   */
  public void afterFlush(ISrvMgrCall[] callList) {
    addMessages();
        
    // propagate event after flush to all ServiceModules
    Iterator it = serviceModules.values().iterator();
    while ( it.hasNext() ) {
      ServiceModule serviceModule = (ServiceModule) it.next();
      serviceModule.onServiceQueueFlushed();
    }
  }

  /**
   * Closes the connection to the backend server.</p>
   * Caution: after connection is closed, each method call on ServiceFacade or GCP objects created from this ServiceFacade
   * could lead to Exceptions, when backend connection is needed explicitly or implicitly!</p>
   * After this call is executed, you should end the application or work wit another ServiceFacade. 
   */
  public void closeConnection() {
    if ( serviceManager != null ) {
      serviceManager.closeConnection();
      serviceManager = null;
    }
  }

  /**
   * @see com.sap.tc.col.client.generic.api.IServiceFacade#getServiceModule(String, String)
   */
  public IServiceModule getServiceModule(String serviceModuleName, String configuration) {
    ServiceModule serviceModule = (ServiceModule) serviceModules.get(serviceModuleName+configuration);
    if ( serviceModule == null ) {
      serviceModule = new ServiceModule(this,serviceManager,serviceManager.getAspectServiceAccess(serviceModuleName));
      serviceModules.put(serviceModuleName+configuration, serviceModule);
    }
    return serviceModule;
  }

  /**
   * @see com.sap.tc.col.client.generic.api.IServiceFacade#getServiceModuleConfigurationNames()
   */
  public String[] getServiceModuleConfigurationNames(String serviceModuleName) {
    return getServiceManager().getServiceModuleConfigurationNames(serviceModuleName);
  }

  /**
   * @see com.sap.tc.col.client.generic.api.IServiceFacade#getServiceModuleDescriptor(String, String)
   */
  public IServiceModuleDescriptor getServiceModuleDescriptor( String serviceModuleName, String configuration) {
    return ((CAFServiceManager)getServiceManager()).getServiceModuleDescriptor(serviceModuleName, configuration);
  }

  /**
   * @see com.sap.tc.col.servicemanager.api.ISrvMgrCallbackOnFlush#beforeFlush()
   */
  public boolean beforeFlush() {
  	beforeFlush(null, null);
	return true;
  }
  
	/* (non-Javadoc)
	 * @see com.sap.tc.col.servicemanager.api.ISrvMgrCallbackOnFlush#beforeFlush(java.lang.String, java.lang.String)
	 */
	public boolean beforeFlush(String arg0, String arg1) {
		// propagate event before flush to all ServiceModules
	   Iterator it = serviceModules.values().iterator();
	   while ( it.hasNext() ) {
		 ServiceModule serviceModule = (ServiceModule) it.next();
		 serviceModule.onServiceQueueWillBeFlushed();
	   }
		return true;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.col.client.generic.api.IServiceFacade#resetLockCache()
	 */
	public void resetLockCache() {
		// TODO Auto-generated method stub

	}

	/* (non-Javadoc)
	 * @see com.sap.tc.col.client.generic.api.IServiceFacade#check()
	 */
	public boolean check() {
		// TODO Auto-generated method stub
		return false;
	}

	/* (non-Javadoc)
	 * @see com.sap.tc.col.client.generic.api.IServiceFacade#getAndClearSystemMessages()
	 */
	public IMessageList getAndClearSystemMessages() {
		ISrvMgrMessage[] sysMsgs = getServiceManager().getAndClearSystemMessages();
		MessageList ml = new MessageList();
		for (int i = 0; i < sysMsgs.length; i++) {
			Message m = new Message(sysMsgs[i]);
			ml.addMessage(m);
		}
		return ml; 
	}
}
