package com.sap.caf.rt.ui.cool.generic;

import java.util.BitSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;

import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.tc.col.client.generic.api.IKey;
import com.sap.tc.col.client.generic.api.IKeyList;
import com.sap.tc.col.client.generic.api.SortingCriteria;
import com.sap.tc.col.client.metadata.api.IKeyAspectDescriptor;
import com.sap.tc.col.client.metadata.api.IStructureDescriptor;
import com.sap.tc.col.edo.IEdoStructure;
import com.sap.tc.col.edo.IEdoTable;
import com.sap.tc.col.servicemanager.api.calls.ISrvMgrSelect;
import com.sap.tc.logging.Location;


/**
 * this class provides helper methods to copy/wrap data from EDO data structures
 * to GCP Aspects, AspectRows, KeyLists and vice versa.
 * @author Helmut Mueller
 */
public class GenericClientUtils {
   
	/** Logging properites for this class */
	private static final String APPLICATION	= GenericClientUtils.class.getName();
	private static final String jARMRequest = AbstractModelClass.jARMReqPrefix+APPLICATION;
	private static final Location logger = Location.getLocation(APPLICATION);
    
  /**
   * helper method which copies Keys to an EdoTable
   * @param inputKeyList the input KeyList
   * @param outTable the target of the copy
   */
  static protected void copyFromKeyListToTable(IKeyList inputKeyList, IEdoTable outTable) {
		final String method = jARMRequest + ":copyFromKeyListToTable(IKeyList, IEdoTable)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		try {
	    IStructureDescriptor descriptor = inputKeyList.getDescriptor().getStructure();
	    
	    for (int i=0; i<inputKeyList.size(); i++) {
	      Key key = (Key) inputKeyList.getKey(i);
	      if ( key.isLocalKey() )
	        continue;
	      outTable.append(1);
	      String[] keyFields = key.getKeyFields();
	      for( int j=0; j<descriptor.size(); j++ ) {
	        outTable.setStringValue(i, j, keyFields[j] );
	      }
	    }
		}
		finally {   
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		}
  }
  
  /**
   * helper method which copies Keys and Fields of changed AspectRows to given parameters 
   * of an Update/Insert field call
   */
  static protected void copyKeysAndFieldsToCall(Aspect aspect, Map changedRows, IEdoTable inFields) {
		final String method = jARMRequest + ":copyKeysAndFieldsToCall(Aspect, Map, IEdoTable)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		try {
	    IStructureDescriptor descriptor = aspect.getDescriptor().getStructure();
	    Iterator it = changedRows.entrySet().iterator();
	    int counter = 0;
	    int noOfKey = 0;
	    while ( it.hasNext() ) {
	      Map.Entry entry = (Map.Entry) it.next();
	      BitSet fields = (BitSet)entry.getValue();
	      for( int i=0; i<descriptor.size(); i++ ) {
	        if ( fields.get(i) ) {
	          inFields.append(1);
	          String fieldName = descriptor.getFieldDescriptor(i).getName();
	          String inKeyAsString = String.valueOf(noOfKey);
	          inFields.setStringValue(counter, "INKEY", inKeyAsString );
	          inFields.setStringValue(counter, "FIELDNAME", fieldName );
	          // inFields.setStringValue(counter, "FIELDVALUE", row.getAttributeAsString(fieldName) );
	          counter++;
	        }
	      }
	      noOfKey++;
	    }   
		}
		finally {   
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		}
  }
  
  /**
   * helper method which copies a Key to an EdoTable to table row with given rowIndex
   * @param inputKeyList the input KeyList
   * @param outTable the target of the copy
   */
  static protected void copyFromKeyToTable(Key key, IEdoTable outTable, int rowIndex) {
		final String method = jARMRequest + ":copyFromKeyToTable(Key, IEdoTable, int)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		try {
	    IStructureDescriptor descriptor = key.getKeyDescriptor().getStructure();
	    
	    outTable.append(1);
	    String[] keyFields = key.getKeyFields();
	    for( int j=0; j<descriptor.size(); j++ ) {
	      outTable.setStringValue(rowIndex, j, keyFields[j] );
	    }
		}
		finally {   
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		}
  }

  /** 
   * extracts a key from the given row of the given table
   * @todo optimize it
   */
  static protected IKey getKeyFromTable(IEdoTable table, IKeyAspectDescriptor keyDescriptor, int row) {
		final String method = jARMRequest + ":getKeyFromTable(IEdoTable, IKeyDescriptor, int)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		try {
	    int nKeyFields = keyDescriptor.getStructure().size();
	    String[] keyFields = new String[nKeyFields];
	    for( int i=0; i<nKeyFields; i++ ) {
	      String fieldName = keyDescriptor.getStructure().getFieldDescriptor(i).getName();
	      int fieldIndex = table.getDescriptor().getFieldIndex(fieldName);
	      keyFields[i] = table.getStringValue(row, fieldIndex);
	    }
	    return new Key(keyDescriptor, keyFields);
		}
		finally {   
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		}
  }

  /**
   * method merges results of a Select call to an Aspect. This select is not done to build
   * the Aspect a new, but to refresh it.
   * we have two cases to consider:
   * <ul>
   * <li> only one AspectRow is to refresh
   * <li> the complete Aspect is to refresh
   * </ul>
   */
  static public IEdoTable mergeResultWithAspect(Aspect aspect, ISrvMgrSelect selectCall) {
		final String method = jARMRequest + ":mergeResultWithAspect(Aspect, ISrvMgrSelect)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		try {
	    IEdoTable inKeys = selectCall.getInKeys();
	    IEdoTable outRecords = selectCall.getOutRecords();
	    if ( aspect.size() > 1 && inKeys.getRecordCount() == 1 ) {
	      // first case: refresh of one row and Aspect has more than one AspectRow
	      IKey key = getKeyFromTable(inKeys, aspect.getDescriptor().getKeyDescriptor(), 0);
	      AspectRow row = (AspectRow) aspect.getAspectRow(key);
	      if ( outRecords != null && outRecords.getRecordCount() == 1 ) {
	        // merge outRecord to aspect data
	        aspect.getAspectData().setTable(new int[] {row.getIndex()}, outRecords, 0);
	        row.setState(Aspect.CLEAN);
	      }
	      else {
	        // remove the AspectRow, which should be refreshed and set state to INVALID
	        row.setState(Aspect.PROCESS_QUEUE_WAITING);
	        aspect.removeAspectRow(row);
	        row.setState(Aspect.INVALID);
	      }
	      // return existing wrapped EDO table
	      return aspect.getAspectData();
	    }
	    // the second case: complete Aspect is to refresh
	    // if number of inkeys == outRecords, we only have to change, 
	    // but only as long as the sort order is the same 
	    // wrapped EDO table
	    if ( inKeys.getRecordCount() != outRecords.getRecordCount() ) {
	      // get Keys of all outRecords
	      HashSet outKeys = new HashSet();
	      for( int i=0; i<outRecords.getRecordCount(); i++ ) {
	        IKey key = getKeyFromTable(outRecords, aspect.getDescriptor().getKeyDescriptor(), i);
	        outKeys.add(key);
	      }
	      // loop over all inKeys
	      for( int i=0; i<inKeys.getRecordCount(); i++ ) {
	        IKey key = getKeyFromTable(inKeys, aspect.getDescriptor().getKeyDescriptor(), i);
	        if ( outKeys.contains(key) )
	          continue;
	        AspectRow row = (AspectRow) aspect.getAspectRow(key);
	        row.setState(Aspect.PROCESS_QUEUE_WAITING);
	        aspect.removeAspectRow(row);
	        row.setState(Aspect.INVALID);
	      } 
	    }
	    return outRecords;
		}
		finally {   
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		}
  }
  
  /**
   * helper method, which returns the unqualified class name for given
   * qualified class name like "com.sap....XXX"
   */
  protected static String getUnqualifiedClassName( String classname ) {
    int index = classname.lastIndexOf(".");
    return classname.substring(index+1);
  }
  /**
   * helper method which copies a Key to an EdoStructure 
   * @param key the input Key
   * @param structure the target of the copy
   */
  protected static void copyFromKeyToStructure(Key key,IEdoStructure structure) {
		final String method = jARMRequest + ":copyFromKeyToStructure(Key, IEdoStructure)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		try {
	    IStructureDescriptor descriptor = key.getKeyDescriptor().getStructure();
	    
	    String[] keyFields = key.getKeyFields();
	    for( int j=0; j<descriptor.size(); j++ ) {
	      structure.setStringValue( j, keyFields[j] );
	    }   
		}
		finally {   
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		}
  }
  
  /**
   * helper method which copies from IEdoStructure to row of a IEdoTable 
   */
  protected static void copyFromEdoStructureToTable(IEdoStructure structure, IEdoTable table, int index) {
		final String method = jARMRequest + ":copyFromEdoStructureToTable(IEdoStructure, IEdoTable, int)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		try {
	    IStructureDescriptor descriptor = structure.getDescriptor();
	    
	    for( int j=0; j<descriptor.size(); j++ ) {
	      table.setValue(index, j, structure.getValue(j) );
	    }   
		}
		finally {   
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		}
  }

  /**
   * helper method, which copies SortingCriteria to IEdoTable
   * @param sortingCriteria
   * @param iEdoTable
   */
  public static void copyFromSortingCriteriaToCall(SortingCriteria sortingCriteria,IEdoTable sortingTable) {
		final String method = jARMRequest + ":copyFromSortingCriteriaToCall(SortingCriteria, IEdoTable)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger);
		try {
	    int noOfCriteria = sortingCriteria.size();
	    if ( noOfCriteria == 0 ) return;
	    
	    sortingTable.append(noOfCriteria);
	    
	    for (int i=0; i<noOfCriteria; i++) {
	      String sortingString = sortingCriteria.getFieldName(i);
	      if ( SortingCriteria.ASCENDING )
	        sortingString += " ASCENDING";
	      else
	        sortingString += " DESCENDING";
	      sortingTable.setStringValue(i, "FIELD", sortingString);
	    }
		}
		finally {   
			CAFPublicLogger.exiting(null, jARMRequest, method, logger);
		}
  }

}
