package com.sap.caf.rt.ui.cool.generic;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.tc.cmi.metadata.CMIModelType;
import com.sap.tc.cmi.metadata.CMISetting;
import com.sap.tc.cmi.metadata.CMISettingDefinition;
import com.sap.tc.cmi.metadata.ICMIModelClassInfo;
import com.sap.tc.cmi.metadata.ICMIModelInfo;
import com.sap.tc.cmi.metadata.ICMIModelObjectCollectionInfo;
import com.sap.tc.col.client.metadata.api.IAspectActionDescriptor;
import com.sap.tc.col.client.metadata.api.IAspectDescriptor;
import com.sap.tc.col.client.metadata.api.IQueryDescriptor;
import com.sap.tc.col.client.metadata.api.IServiceModuleDescriptor;
import com.sap.tc.col.client.metadata.api.IStructureDescriptor;
import com.sap.tc.logging.Location;

/**
 * realizes meta data of a COL model, the class is the factory for <code>ColModelClassInfo</code> instances.</p>
 * @author Helmut Mueller, Stephan Ritter
 */
public class ColModelInfo implements ICMIModelInfo {

	/** Logging properites for this class */
	private static final String APPLICATION	= ColModelInfo.class.getName();
	private static final String jARMRequest = AbstractModelClass.jARMReqPrefix+APPLICATION;
	private static final Location logger = Location.getLocation(APPLICATION);
  
  /**
   * constructor, which only takes model name as parameter
   * @param name the name of the model
   */
  protected ColModelInfo( IServiceModuleDescriptor descriptor ) {
    this.descriptor = descriptor;
    createModelClassInfos( descriptor );
  }

  /**
   * constructor, which only takes model name as parameter
   * @param name the name of the model
   */
  protected ColModelInfo(CAFRepositoryManager repositoryManager, IServiceModuleDescriptor descriptor ) {
    this.descriptor = descriptor;
    this.repositoryManager = repositoryManager;
    createModelClassInfos( descriptor );
  }


  protected CAFRepositoryManager repositoryManager;

  /**
   * the wrapped ServiceModuleDescriptor
   */
  private IServiceModuleDescriptor descriptor;
 
  /**
   * map of all existing meta data of model classes
   */
  private Map modelClassInfos = new HashMap();

  /**
   * returns a collection of all model classes meta data
   */
  public Collection getModelClassInfos() {
    return modelClassInfos.values();
  }

  /**
   * returns an iterator for all model classes meta data
   */
  public Iterator iterateModelClassInfos() {
    return getModelClassInfos().iterator();
  }

  /**
   * returns meta data of model class with given name
   */
  public ICMIModelClassInfo getModelClassInfo(String name) {
    return (ICMIModelClassInfo) modelClassInfos.get(name);
  }

  /**
   * map of all existing aspect infos 
   */
  private final Map modelObjectCollectionInfos = new HashMap();
  
  /**
   * Returns the <code>ColModelObjectCollectionInfo</code> for the given <code>AspectDescriptor</code>.
   * If no info exists yet, one is created and cached. THere will always be at most one 
   * collection info for a single aspect descriptor.
   * @param aspectDescriptor the meta data of an <code>Aspect</code>
   * @return ICMIModelObjectCollectionInfo the created meta data wrapper
   */
  protected ICMIModelObjectCollectionInfo getOrCreateModelObjectCollectionInfo(IAspectDescriptor aspectDescriptor) {
    ICMIModelObjectCollectionInfo info = (ICMIModelObjectCollectionInfo) modelObjectCollectionInfos.get(aspectDescriptor.getName());
    if ( info == null ) {
      info = new ColModelObjectCollectionInfo(this, aspectDescriptor);
      modelObjectCollectionInfos.put(aspectDescriptor.getName(), info);
    }
    return info;
  }    
  
  /**
   * factory method to create <code>ColModelClassInfo</code> from given <code>AspectDescriptor</code>
   * @param aspectDescriptor the meta data of a <COL> <code>Aspect</code>
   * @return ICMIModelClassInfo the created meta data wrapper
   */
  protected ICMIModelClassInfo getOrCreateModelClassInfo(IAspectDescriptor aspectDescriptor) {
    ICMIModelClassInfo modelClassInfo = getModelClassInfo(ColModelClassInfo.ASPECT + aspectDescriptor.getName());
    if (modelClassInfo == null) {
      modelClassInfo = new ColModelClassInfo(aspectDescriptor, this);
      modelClassInfos.put(modelClassInfo.getName(), modelClassInfo);
    }  
    return modelClassInfo; 
  }  
  
  /**
   * factory method to create ICMIModelClassInfo from given <code>QueryDescriptor</code>
   * @param queryDescriptor the meta data of a <COL> <code>Query</code>
   * @return ICMIModelClassInfo the created meta data wrapper
   */
  protected ICMIModelClassInfo getOrCreateModelClassInfo(IQueryDescriptor queryDescriptor) {
    ICMIModelClassInfo modelClassInfo = getModelClassInfo(ColModelClassInfo.QUERY + queryDescriptor.getName());
    if (modelClassInfo == null) {
      modelClassInfo = new ColModelClassInfo(queryDescriptor, this);
      modelClassInfos.put(modelClassInfo.getName(), modelClassInfo);
    }  
    return modelClassInfo; 
  } 
  
  /**
   * factory method to create ICMIModelClassInfo from given StructureDescriptor
   * @param structureDescriptor the meta data of a <COL> <code>Structure</code>
   * @return ICMIModelClassInfo the created meta data wrapper
   */
  protected ICMIModelClassInfo getOrCreateModelClassInfo(IStructureDescriptor structureDescriptor) {
    ICMIModelClassInfo modelClassInfo = getModelClassInfo(ColModelClassInfo.STRUCTURE + structureDescriptor.getName());
    if (modelClassInfo == null) {
      modelClassInfo = new ColModelClassInfo(structureDescriptor, this);
      modelClassInfos.put(modelClassInfo.getName(), modelClassInfo);
    }  
    return modelClassInfo; 
  } 
  
  /**
   * factory method to create ICMIModelClassInfo from given AspectActionDescriptor
   * @param actionDescriptor the meta data of a <COL> <code>Action</code>
   * @return ICMIModelClassInfo the created meta data wrapper
   */
  protected ICMIModelClassInfo getOrCreateModelClassInfo(IAspectActionDescriptor actionDescriptor) {
    ICMIModelClassInfo modelClassInfo = getModelClassInfo(ColModelClassInfo.ACTION + actionDescriptor.getName());
    if (modelClassInfo == null) {
      modelClassInfo = new ColModelClassInfo(actionDescriptor, this);
      modelClassInfos.put(modelClassInfo.getName(), modelClassInfo);
    }  
    return modelClassInfo; 
  }   

  /**
   * returns <code>false</code>, cause modelRelationInfos are not accessible at runtime. 
   * @return boolean <code>false</code>, cause modelRelationInfos are not accessible at runtime. 
   */
  public boolean supportsModelRelationInfos() {
    return false;
  }
  /**
   * returns <code>null</code>, cause the model implementation doesn't support relation infos.
   * @return Collection <code>null</code>, cause the model implementation doesn't support relation infos.
   */
  public Collection getModelRelationInfos() {
    return null;
  }
  
  /**
   * returns <code>null</code>, cause the model
   * implementation doesn't support relation infos.
   * @return Iterator <code>null</code>, cause the model implementation doesn't support relation infos.
   */
  public Iterator iterateModelRelationInfos() {
    return null;
  }
  
  /**
   * returns <code>null</code>, cause 
   * the model implementation doesn't support relation infos.
   * @return Collection <code>null</code>, cause the model implementation doesn't support relation infos.
   */
  public Collection getModelRelationInfo(String name) {
    return null;
  }
  
  /**
   * returns TYPE_GENERIC,the model type for a COL model
   */
  public CMIModelType getModelType() {
    return TYPE_GENERIC;
  }

  /**
   * returns the name of the model
   */
  public String getName() {
    return descriptor.getName();
  }

  /**
   * @see com.sap.tc.cmi.metadata.ICMIAbstractInfo#getSetting(CMISettingDefinition)
   * throws <code>UnsupportedOperationException</code>
   */
  public CMISetting getSetting(CMISettingDefinition settingDef) {
    throw new java.lang.UnsupportedOperationException();
  }

  /**
   * @see com.sap.tc.cmi.metadata.ICMIAbstractInfo#setSetting(CMISettingDefinition, String)
   * throws <code>UnsupportedOperationException</code>
   */
  public void setSetting(CMISettingDefinition settingDef, String value) {
    throw new java.lang.UnsupportedOperationException();
  }

  /**
   * @see com.sap.tc.cmi.metadata.ICMIAbstractInfo#getSettings()
   * throws <code>UnsupportedOperationException</code>
   */
  public Map getSettings() {
    throw new java.lang.UnsupportedOperationException();
  }

  /**
   * @see com.sap.tc.cmi.metadata.ICMIAbstractInfo#supports(String)
   * throws <code>UnsupportedOperationException</code>
   */
  public boolean supports(String option) {
    throw new java.lang.UnsupportedOperationException();
  }

  /**
   * @see com.sap.tc.cmi.metadata.ICMIAbstractInfo#supportedOptions()
   * throws <code>UnsupportedOperationException</code>
   */
  public Collection supportedOptions() {
    throw new java.lang.UnsupportedOperationException();
  }

  /**
   * @see com.sap.tc.cmi.metadata.ICMIAbstractInfo#addSupportedOption(String)
   * throws <code>UnsupportedOperationException</code>
   */
  public void addSupportedOption(String option) {
    throw new java.lang.UnsupportedOperationException();
  }
  
  /**
   * helper method, which creates all <code>ModelClassInfo</code> classes for
   * this <code>Modelnfo</code> class.<p>
   */
  private void createModelClassInfos(IServiceModuleDescriptor descriptor) {
		final String method = jARMRequest + ":createModelClassInfos(IServiceModuleDescriptor)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger, 2);
		try {
	    IQueryDescriptor[] queryDescriptors = descriptor.getQueryDescriptors();
	    for ( int i=0; i<queryDescriptors.length; i++ ) {
	      createModelClassesForQuery( queryDescriptors[i] );
	    }
	    IAspectDescriptor[] aspectDescriptors = descriptor.getAspectDescriptors(); 
	    for ( int i=0; i<aspectDescriptors.length; i++ ) {
	      createModelClassesForAspect( aspectDescriptors[i] );
	    }  
		}
		finally {      
			CAFPublicLogger.exiting(null, jARMRequest, method, logger, 2);
		}
  }

  /**
   * Method createModelClassesForAspect.
   * @param aspectDescriptor
   */
  private void createModelClassesForAspect(IAspectDescriptor aspectDescriptor) {
		final String method = jARMRequest + ":createModelClassesForAspect(IAspectDescriptor)";
		CAFPublicLogger.entering(null, jARMRequest, method, logger, 2);
		try {
	    // first create model info class for Aspect itself
	    ICMIModelClassInfo classInfo = getOrCreateModelClassInfo(aspectDescriptor);
	    // ... then for Actions
	    IAspectActionDescriptor[] actionDescriptors = aspectDescriptor.getAspectActionDescriptors();
	    for ( int i=0; i<actionDescriptors.length; i++ ) {
	      IAspectActionDescriptor aspectActionDescriptor = actionDescriptors[i];
	      classInfo = getOrCreateModelClassInfo(aspectActionDescriptor);
	      if ( aspectActionDescriptor.hasInputParameters() )
	        classInfo = getOrCreateModelClassInfo( aspectActionDescriptor.getInputParameters() );
	    } 
		}
		finally {      
			CAFPublicLogger.exiting(null, jARMRequest, method, logger,2 );
		}
  }

  /**
   * Method createModelClassesForQuery.
   * @param queryDescriptor
   */
  private void createModelClassesForQuery(IQueryDescriptor queryDescriptor) {
    // first create model info class for Query itself
    ICMIModelClassInfo classInfo = getOrCreateModelClassInfo(queryDescriptor);
    // .. then for inputParameter
    if ( queryDescriptor.hasInputParameters() )
      classInfo = getOrCreateModelClassInfo(queryDescriptor.getInputParameters());
    // .. then for result
    classInfo = getOrCreateModelClassInfo(queryDescriptor.getResultAspectDescriptor());
  }
}
