/*-----------------------------------------------------------------------------------
Copyright (c)2003 Borland Software Corporation. Patents pending. All rights reserved.
-----------------------------------------------------------------------------------*/

package user_interface;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.math.BigDecimal;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.JToolBar;
import javax.swing.SwingConstants;
import javax.swing.WindowConstants;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumn;

import problem_domain.CashSale;
import problem_domain.CashSaleDetail;
import problem_domain.InsuffPaymentException;
import problem_domain.ProductDesc;
import server.DMServer;

/**
 * This is a sample that was based on the JMCU CashSale component built into
 * the Together tool. This presents a simple GUI and
 * shows how we can tie it into the PD classes.
 * @subtitle The std cashier UI
 * @stereotype ui-component
 * @author fred*/
public class POSFrame extends JFrame
{
    /** Use this to trigger the database on/off */
    //public static boolean USE_DB = false;  // 'false' turns DB functions OFF
    public static boolean USE_DB = false;  // 'true' turns DB functions ON

    /** Problem Domain Object */
    private CashSale currentSale = new CashSale();

    /** Database handle */
    private DMServer dmServer = null;

    /** List of store items */
    private ProductDesc[] products;

    /** number of products for sale */
    private int numProducts = 0;

    /** List of Cashiers */
    private String[] cashiers;

    /** number of Cashiers */
    private int numCashiers = 0;

    /** Sale Detail Table Column Header */
    private final String[] colNames = {"Item", "Name", "Unit", "Qty", "Price"};

    final static int ADDED_DETAIL = 0;
    final static int REMOVED_DETAIL = 1;

	/** Format helper */
    SimpleDateFormat dateFormat = new SimpleDateFormat ("yyyy.MM.dd");

	/** Format helper */
    NumberFormat currencyFormat = NumberFormat.getCurrencyInstance();

    JMenuBar menuBar1 = new JMenuBar();
    JMenu menuFile = new JMenu();
    JMenuItem menuFileExit = new JMenuItem();
    JMenu menuView = new JMenu();
    JMenuItem menuViewListAll = new JMenuItem();
    JMenu menuHelp = new JMenu();
    JMenuItem menuHelpAbout = new JMenuItem();
    JToolBar toolBar = new JToolBar();
    ImageIcon imageHelp;
    JLabel statusBar = new JLabel();
    BorderLayout borderLayout1 = new BorderLayout();
    JScrollPane jScrollPaneDetails = new JScrollPane();
    JTable jTableSaleDetails = new JTable(new TableDataModel());
    JPanel jPanel1 = new JPanel();
    BorderLayout borderLayout2 = new BorderLayout();
    JPanel jPanelScan = new JPanel();
    JButton jButtonScan = new JButton();
    JTextField jTextFieldItem = new JTextField();
    JPanel jPanelDetails = new JPanel(new GridBagLayout());
    JButton jButtonNewSale = new JButton();
    JButton jButtonCancel = new JButton();
    JPanel jPanelCompleteSale = new JPanel();
    JButton jButtonPayment = new JButton();
    JButton jButtonTotal = new JButton();
    BorderLayout borderLayout3 = new BorderLayout();
    JTextField jTextFieldSubTotal = new JTextField();
    GridLayout gridLayout1 = new GridLayout();
    JLabel jLabel1 = new JLabel();
    JLabel jLabel2 = new JLabel();
    JTextField jTextFieldTax = new JTextField();
    JLabel jLabel3 = new JLabel();
    JTextField jTextFieldPayment = new JTextField();
    JTextField jTextFieldTotal = new JTextField();
    JTextField jTextFieldChange = new JTextField();
    JLabel jLabel4 = new JLabel();
    JLabel jLabelCashier = new JLabel();
    JLabel jLabel6 = new JLabel();
    JLabel jLabelTimeStamp = new JLabel();
    JLabel jLabel8 = new JLabel();

    /** Construct the frame */
    public POSFrame()
    {
        setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE); //do
        // not hide the frame, maybe the user will press "Cancel"
        addWindowListener(
            new WindowAdapter()
            {
                public void windowClosing(WindowEvent e)
                {
                    fileExit_actionPerformed(null);
                }
            });
        try
        {
            // Not the best place to start database, but it works...
            initDatabase();

			// Set up the Cashier list
            setUpCashiers();

            // Get the Cash Register up and running with current product list
            setUpProducts();

			// initialize widget stuff
            initialize();
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }  // END POSFrame()

    /** Set up list of cashiers */
    private void setUpCashiers()
    {
        System.out.println("Loading Cashier List");

        cashiers = new String[10];

        cashiers[0] = "Jim";
        cashiers[1] = "Lucy";
        cashiers[2] = "Steve";
        cashiers[3] = "Sarah";
        cashiers[4] = "Jon";
        cashiers[5] = "Buddy";
        cashiers[6] = "Bettie";
        cashiers[7] = "Sue";
        cashiers[8] = "John";
        cashiers[9] = "Ted";

        numCashiers = cashiers.length;
    }  // END setUpCashiers()

    /** Set up the product "database" */
    private void setUpProducts()
    {
        System.out.println("Loading Local Product Database");

        // Eventually, get this from persistent storage...
        products = new ProductDesc[10];
        products[0] = new ProductDesc("1", "Pepsi 24-pack", "Pepsi 24", new BigDecimal(3.99));
        products[1] = new ProductDesc("2", "Lays Ridges", "Lays", new BigDecimal(1.99));
        products[2] = new ProductDesc("3", "Vienna Sausages", "Vienna Sausages",
            new BigDecimal(2.99));
        products[3] = new ProductDesc("4", "White Popcorn", "White Popcorn",
            new BigDecimal(1.29));
        products[4] = new ProductDesc("5", "Soy Burgers", "Soy Burger", new BigDecimal(5.99));
        products[5] = new ProductDesc("6", "Cat Chow", "Cat Chow", new BigDecimal(9.99));
        products[6] = new ProductDesc("7", "Puppy Chow", "Puppy Chow", new BigDecimal(12.99));
        products[7] = new ProductDesc("8", "Finch Food", "Finch Food", new BigDecimal(1.59));
        products[8] = new ProductDesc("9", "Rice Krispies", "Rice Krispies",
            new BigDecimal(3.29));
        products[9] = new ProductDesc("10", "Fruit Loops", "Fruit Loops",
            new BigDecimal(3.49));

        numProducts = products.length;
    }  // END setUpProducts()

    //Component initialization


    private void initialize() throws Exception
    {
        System.out.println("setting up UI");
        this.getContentPane().setLayout(borderLayout1);
        this.setSize(new Dimension(459, 396));
        this.setTitle("Point Of Sale");
        statusBar.setText(" ");
        menuFile.setText("File");
        menuFile.setMnemonic('F');
        menuFileExit.setText("Exit");
        menuFileExit.setMnemonic('x');
        menuFileExit.addActionListener(
            new ActionListener()
            {
                public void actionPerformed(ActionEvent e)
                {
                    fileExit_actionPerformed(e);
                }
            });

        menuView.setText("View");
        menuView.setMnemonic('V');
        menuViewListAll.setText("List all sales");
        menuViewListAll.setMnemonic('L');
        menuViewListAll.addActionListener(
            new ActionListener()
            {
                public void actionPerformed(ActionEvent e)
                {
                    viewListAll_actionPerformed(e);
                }
            });

        menuHelp.setText("Help");
        menuHelpAbout.setText("About");
        menuHelpAbout.addActionListener(
            new ActionListener()
            {
                public void actionPerformed(ActionEvent e)
                {
                    helpAbout_actionPerformed(e);
                }
            });
        jPanel1.setLayout(borderLayout2);
        jButtonScan.setToolTipText("Scan next item");
        jButtonScan.setMnemonic('S');
        jButtonScan.setText("Scan");
        jButtonScan.addActionListener(
            new java.awt.event.ActionListener()
            {
                public void actionPerformed(ActionEvent e)
                {
                    jButtonScan_actionPerformed(e);
                }
            });
        jTextFieldItem.setPreferredSize(new Dimension(182, 21));
        jTextFieldItem.setText("<item>");
        jPanelScan.setBorder(BorderFactory.createRaisedBevelBorder());
        jTableSaleDetails.setBorder(BorderFactory.createLineBorder(Color.black));
        jButtonNewSale.setToolTipText("Start a new sale");
        jButtonNewSale.setMnemonic('N');
        jButtonNewSale.setText("New Sale");
        jButtonNewSale.addActionListener(
            new java.awt.event.ActionListener()
            {
                public void actionPerformed(ActionEvent e)
                {
                    jButtonNewSale_actionPerformed(e);
                }
            });
        jButtonCancel.setToolTipText("Cancel current sale");
        jButtonCancel.setMnemonic('A');
        jButtonCancel.setText("Cancel");
        jButtonCancel.addActionListener(
            new java.awt.event.ActionListener()
            {
                public void actionPerformed(ActionEvent e)
                {
                    jButtonCancel_actionPerformed(e);
                }
            });
        jButtonPayment.setFont(new java.awt.Font("Dialog", 1, 12));
        jButtonPayment.setToolTipText("Enter payment amount");
        jButtonPayment.setMnemonic('P');
        jButtonPayment.setText("Payment");
        jButtonPayment.addActionListener(
            new java.awt.event.ActionListener()
            {
                public void actionPerformed(ActionEvent e)
                {
                    jButtonPayment_actionPerformed(e);
                }
            });
        jButtonTotal.setFont(new java.awt.Font("Dialog", 1, 12));
        jButtonTotal.setToolTipText("Calculate Total");
        jButtonTotal.setMnemonic('T');
        jButtonTotal.setText("Total");
        jButtonTotal.addActionListener(
            new java.awt.event.ActionListener()
            {
                public void actionPerformed(ActionEvent e)
                {
                    jButtonTotal_actionPerformed(e);
                }
            });
        GridBagConstraints gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.fill = GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        jTextFieldSubTotal.setText("0.00");
        jTextFieldSubTotal.setHorizontalAlignment(SwingConstants.RIGHT);
        jPanelCompleteSale.setLayout(gridLayout1);
        gridLayout1.setColumns(2);
        gridLayout1.setHgap(3);
        gridLayout1.setRows(5);
        jLabel1.setFont(new java.awt.Font("Dialog", 1, 12));
        jLabel1.setHorizontalAlignment(SwingConstants.RIGHT);
        jLabel1.setText("SUBTOTAL:");
        jLabel2.setFont(new java.awt.Font("Dialog", 1, 12));
        jLabel2.setHorizontalAlignment(SwingConstants.RIGHT);
        jLabel2.setText("Tax:");
        jLabel3.setText("       ");
        jTextFieldTax.setText("0.00");
        jTextFieldTax.setHorizontalAlignment(SwingConstants.RIGHT);
        jTextFieldPayment.setText("0.00");
        jTextFieldPayment.setHorizontalAlignment(SwingConstants.RIGHT);
        jTextFieldTotal.setText("0.00");
        jTextFieldTotal.setHorizontalAlignment(SwingConstants.RIGHT);
        jTextFieldChange.setText("0.00");
        jTextFieldChange.setHorizontalAlignment(SwingConstants.RIGHT);
        jLabel4.setText("       ");
        jLabelCashier.setText("       ");
        jLabel6.setText("       ");
        jLabelTimeStamp.setText("       ");
        jLabel8.setText("Change:");
        jLabel8.setFont(new java.awt.Font("Dialog", 1, 12));
        jLabel8.setHorizontalAlignment(SwingConstants.RIGHT);
        jScrollPaneDetails.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        jScrollPaneDetails.setBorder(BorderFactory.createLineBorder(Color.black));
        toolBar.add(jButtonNewSale);
        toolBar.add(jButtonCancel);
        menuFile.add(menuFileExit);
        menuView.add(menuViewListAll);
        menuHelp.add(menuHelpAbout);
        menuBar1.add(menuFile);
        menuBar1.add(menuView);
        menuBar1.add(menuHelp);
        this.setJMenuBar(menuBar1);
        this.getContentPane().add(toolBar, BorderLayout.NORTH);
        this.getContentPane().add(statusBar, BorderLayout.SOUTH);
        this.getContentPane().add(jPanel1, BorderLayout.CENTER);
        jPanel1.add(jPanelScan, BorderLayout.NORTH);
        jPanelScan.add(jButtonScan, null);
        jPanelScan.add(jTextFieldItem, null);
        jPanel1.add(jPanelDetails, BorderLayout.CENTER);
        jPanelDetails.add(jScrollPaneDetails, gridBagConstraints);
        jScrollPaneDetails.getViewport().add(jTableSaleDetails, null);
        jPanel1.add(jPanelCompleteSale, BorderLayout.SOUTH);
        jPanelCompleteSale.add(jLabel3, null);
        jPanelCompleteSale.add(jLabel1, null);
        jPanelCompleteSale.add(jTextFieldSubTotal, null);
        jPanelCompleteSale.add(jLabel4, null);
        jPanelCompleteSale.add(jLabel2, null);
        jPanelCompleteSale.add(jTextFieldTax, null);
        jPanelCompleteSale.add(jLabel6, null);
        jPanelCompleteSale.add(jButtonTotal, null);
        jPanelCompleteSale.add(jTextFieldTotal, null);
        jPanelCompleteSale.add(jLabelCashier, null);
        jPanelCompleteSale.add(jButtonPayment, null);
        jPanelCompleteSale.add(jTextFieldPayment, null);
        jPanelCompleteSale.add(jLabelTimeStamp, null);
        jPanelCompleteSale.add(jLabel8, null);
        jPanelCompleteSale.add(jTextFieldChange, null);
        // Initialize Detail Table
        initDetailTable();
        // Initialize widgets with data
        loadGUIWithPD();
        // Keep buttons properly grayed out at the start
        initButtons();
        this.toolBar.getRootPane().setDefaultButton(jButtonScan);
    }

    private void initDatabase()
    {
        if ( POSFrame.USE_DB )
        {
	        try
	        {
	            dmServer = new DMServer();
				currentSale.setDmServer(dmServer);
	        }
	        catch( java.sql.SQLException sqe )
	        {
	            String msg = new StringBuffer()
	                .append("Have you defined the data source properly?")
	                .append("\n")
	                .append(sqe.getMessage())
	                .toString();
	            JOptionPane.showMessageDialog(null, msg,
	                "Failed Database Initialization", JOptionPane.ERROR_MESSAGE);
	        }
	        catch( ClassNotFoundException cnfe )
	        {
	            JOptionPane.showMessageDialog(null, cnfe.getMessage(),
	                "Failed to Load Database Driver", JOptionPane.ERROR_MESSAGE);
	        }
        }  // ENDIF (USE_DB)
    }  // END initDatabase()

    /** Keep buttons properly grayed out at the start. */
    private void initButtons()
    {
        jButtonNewSale.setEnabled(true);
        jButtonCancel.setEnabled(false);
        jButtonScan.setEnabled(true);
        jButtonPayment.setEnabled(false);
        jButtonTotal.setEnabled(false);
    } // END initButtons()

    /** Load the UI widgets with "data" from the buisness object. */
    private void loadGUIWithPD()
    {
        jTextFieldItem.setText("");
        jLabelTimeStamp.setText(DateFormat.getDateInstance().format(currentSale.getTime()));
        jLabelCashier.setText(cashiers[(int)(Math.random() * numCashiers)]);
        jTextFieldSubTotal.setText("");
        jTextFieldTax.setText("");
        jTextFieldTotal.setText("");
        jTextFieldPayment.setText("");
        jTextFieldChange.setText("");
        jTextFieldChange.setForeground(Color.black);
    } // END loadGUIWithPD()


    private void initDetailTable()
    {

        jTableSaleDetails.setPreferredScrollableViewportSize(new Dimension(300, 70));

        TableColumn column = null;
        // {"Item", "Name", "Unit", "Qty", "Price"};
        for (int i = 0; i < jTableSaleDetails.getColumnCount(); ++i)
        {
            column = jTableSaleDetails.getColumnModel().getColumn(i);
            switch (i)
            {
                case 0: // Item
                    column.setPreferredWidth(20);
                    break;
                case 1: // Name
                    column.setPreferredWidth(100);
                    break;
                case 2: // Unit
                    column.setPreferredWidth(30);
                    break;
                case 3: // Qty
                    column.setPreferredWidth(20);
                    break;
                case 4: // Price
                    column.setPreferredWidth(30);
                    break;
                default:
                    column.setPreferredWidth(50);
                    break;
            } // ENDSWITCH (col)
        } // ENDFOR (i)
    }

    //File | Exit action performed
    public void fileExit_actionPerformed(ActionEvent e)
    {
        if (preCancelCheck() == JOptionPane.OK_OPTION)
        {
            System.exit(0);
        }
    }

    //View | ListAll action performed
    public void viewListAll_actionPerformed(ActionEvent e)
    {
        listAllSales();
    }

    //Help | About action performed
    public void helpAbout_actionPerformed(ActionEvent e)
    {
        POSFrame_AboutBox dlg = new POSFrame_AboutBox(this);
        Dimension dlgSize = dlg.getPreferredSize();
        Dimension frmSize = getSize();
        Point loc = getLocation();
        dlg.setLocation((frmSize.width - dlgSize.width) / 2 + loc.x,
            (frmSize.height - dlgSize.height) / 2 + loc.y);
        dlg.setModal(true);
        dlg.show();
    }
    //=================================================


    private int preCancelCheck()
    {
        if (currentSale.isPending())
        {
            return JOptionPane.showConfirmDialog(null,
                "OK to Cancel Current Sale?", "Sale Pending",
                JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);
        }
        else
        {
            return JOptionPane.OK_OPTION;
        }
    } // END preCancelCheck()

    /** User has decided to cancel the scale. Start a new sale. */
    private void cancelSale()
    {
        newSale();

    } // END cancelSale()

    /**
     * Start off with a fresh CashSale object for the transaction.
     * It will reflect the current timestamp and have no details.
     */
    private void newSale()
    {
        if (!currentSale.isCompleted())
        {
            if (preCancelCheck() != JOptionPane.YES_OPTION)
            {
                return;
            }
        }

		// Initialize current sale instance
        currentSale.clearValues();
        TableDataModel model = (TableDataModel)jTableSaleDetails.getModel();
        model.dataChanged(REMOVED_DETAIL, 0);

        // Reset widgets
        loadGUIWithPD();
        initButtons();
    } // END newSale()

    /** Accept amount of cash paid in, compute change owed. */
    private void makePayment()
    {
		// Get user input. Note: if user enters nothing or cancels, we can
		// simply sit there and do nothing.
		String paymentValue = JOptionPane.showInputDialog("Please enter payment");
		if ( paymentValue != null )
        {
            paymentValue = removeIllegalChars(paymentValue);
        }
        else
        {
            JOptionPane.showMessageDialog(null, "You entered no payment?!",
                "No Payment", JOptionPane.INFORMATION_MESSAGE);
            return;
        }

        try
        {
	        // See if input is valid number
            NumberFormat numberFormat = NumberFormat.getInstance();
            numberFormat.setMinimumFractionDigits(2);
            double n = numberFormat.parse(paymentValue).doubleValue();
            // Compute change due
            BigDecimal change = currentSale.makeCashSale(new BigDecimal(paymentValue));
            jTextFieldChange.setText(currencyFormat.format(change));
            jTextFieldChange.setForeground(Color.red);
            jTextFieldPayment.setText(currencyFormat.format(new BigDecimal(paymentValue)));

            // Save the sale...
            if ( POSFrame.USE_DB)
            {
                currentSale.save();
            }

            // Once payment is made, you can no
            // longer cancel or hit payment.
            jButtonPayment.setEnabled(false);
            jButtonCancel.setEnabled(false);
            this.getRootPane().setDefaultButton(jButtonNewSale);
        }
        catch (java.sql.SQLException sqe)
        {
            JOptionPane.showMessageDialog(null, sqe.getMessage(),
                "Save Error", JOptionPane.ERROR_MESSAGE);
        }
        catch (ParseException pe)
        {
            JOptionPane.showMessageDialog(null, pe.getMessage(),
                "Illegal Text Format", JOptionPane.ERROR_MESSAGE);
            makePayment();
        }
        catch (InsuffPaymentException ipe)
        {
            JOptionPane.showMessageDialog(null, ipe.getMessage(),
                "Insufficient Payment", JOptionPane.ERROR_MESSAGE);
        }

    } // END makePayment()

    /** Total all the items in the sale, add in the tax. */
    private void totalSale()
    {
        jTextFieldSubTotal.setText(currencyFormat.format(currentSale.calcSubtotal()));
        jTextFieldTax.setText(currencyFormat.format(currentSale.calcTax()));
        jTextFieldTotal.setText(currencyFormat.format(currentSale.calcTotal()));
        // Once you total the sale, you can
        // then make a payment. But we'll disable
        // the ability to scan in more or re-calc
        // total (maybe a bit unrealistic).
        jButtonPayment.setEnabled(true);
        jButtonTotal.setEnabled(false);
        jButtonScan.setEnabled(false);
    } // END totalSale()

    /** Use this to simulate scanning items */
    private void scan()
    {
        int pseudoIndex = (int)(Math.random() * numProducts);
        ProductDesc prod = products[pseudoIndex];
        jTextFieldItem.setText(prod.getDescription());
        CashSaleDetail detail = new CashSaleDetail(prod);
        // Now add it to the current sale
        currentSale.addSaleDetail(detail);
        // And notify the table
        addDetailToTable();
        jButtonCancel.setEnabled(true);
        jButtonTotal.setEnabled(true);
    } // END scan()

    /**
     * List all sales.
     * This should be done using a Table Model...
     * And, it should be done to support numerous data presentations,
     * as there will likely be many more than just "list all."
     */
    private void listAllSales()
    {
    	if ( POSFrame.USE_DB )
        {
	        Vector history = currentSale.listCashSales();
	        int nRows = history.size();
	
	        JFrame list = new JFrame( "List All" );

			// Should really define constants for the column name and numbers...
	        String[] tableColumnNames = { "Date", "Total", "Subtotal", "Tax" };
			Object[][] data = new Object[nRows+1][4];

            // Reusable reference
	        CashSale aSale = null;

            // Tally holders for a summary row.
			BigDecimal total = new BigDecimal("0.0");
			BigDecimal sumTotal = new BigDecimal("0.0");
	        BigDecimal sumSubtotal = new BigDecimal("0.0");
	        BigDecimal sumTax = new BigDecimal("0.0");

			// Loop through each row in the result set
            // to build table for list view
	        for ( int i = 0; i < nRows; ++i )
	        {
	            aSale = (CashSale)history.elementAt(i);
	            data[i][0] = dateFormat.format( aSale.getTime() );
                // Could probably refactor out the "total" value
                data[i][1] = currencyFormat.format( aSale.getSubtotal().add(aSale.getTax()) );
	            data[i][2] = currencyFormat.format( aSale.getSubtotal() );
	            data[i][3] = currencyFormat.format( aSale.getTax() );
	
				sumTotal = sumTotal.add( aSale.getSubtotal().add(aSale.getTax()) );
	            sumSubtotal = sumSubtotal.add( aSale.getSubtotal() );
	            sumTax = sumTax.add( aSale.getTax() );
	        }  // ENDFOR (i)
	
	        // Add summary data
	        data[nRows][0] = "SUMMARY";
	        data[nRows][1] = currencyFormat.format( sumTotal );
	        data[nRows][2] = currencyFormat.format( sumSubtotal );
	        data[nRows][3] = currencyFormat.format( sumTax );
	
	        JScrollPane jScrollPaneDetails = new JScrollPane();
	        JTable jTableSaleHistory = new JTable( data, tableColumnNames );
	        jTableSaleHistory.setBorder(BorderFactory.createLineBorder(Color.black));
			jTableSaleHistory.setPreferredScrollableViewportSize( new Dimension(300,200));
			jTableSaleHistory.setColumnSelectionAllowed(false);
	        jTableSaleHistory.setShowVerticalLines(false);
	        jScrollPaneDetails.getViewport().add(jTableSaleHistory, null);
			list.getContentPane().add(jScrollPaneDetails);
	
			// Now display the frame...
	        list.setVisible(true);
	        list.pack();
        }  // ENDIF (USE_DB)
        else
        {
            JOptionPane.showMessageDialog(null,
                "To use this feature, you need to enable the database. See info.",
                "No Database Functionality",
                JOptionPane.INFORMATION_MESSAGE);
        }
    }  // END listAllSales()

    /** Update the table model to update the table UI. */
    private void addDetailToTable()
    {
        int index = currentSale.getDetailList().size() - 1;
        TableDataModel model = (TableDataModel)jTableSaleDetails.getModel();
        model.dataChanged(ADDED_DETAIL, index);
    } // END addDetailToTable()

    /** Helper function to strip chars from currency input widget. */
    private String removeIllegalChars(String string)
    {
        StringBuffer sb = new StringBuffer(string);
        for (int i = 0; i < sb.length(); i++)
            if (sb.charAt(i) == ',')
                sb.setCharAt(i, '.');
        return sb.toString();
    } // END removeIllegalChars()


/* =======================
 *       BUTTON EVENTS
 * ======================= */

    void jButtonNewSale_actionPerformed(ActionEvent e)
    {
        newSale();
    }

    void jButtonCancel_actionPerformed(ActionEvent e)
    {
        cancelSale();
    }

    void jButtonPayment_actionPerformed(ActionEvent e)
    {
        makePayment();
    }

    void jButtonTotal_actionPerformed(ActionEvent e)
    {
        totalSale();
    }

    void jButtonScan_actionPerformed(ActionEvent e)
    {
        scan();
    }

//=================================================
    /**
     * Create a model of the data. NOTE: We could have built our own local
     * vector to store the sale detail objects, but we are using the GUI
     * object's own instance of a CashSale object to access the information
     * pertaining to its collection of SaleDetails.*/
    class TableDataModel extends AbstractTableModel
    {
        // These 3 methods always need to be implemented.
        public int getColumnCount()
        {
            return colNames.length;
        }

        public int getRowCount()
        {
            Vector detailList = currentSale.getDetailList();
            if (detailList != null)
            {
                return detailList.size();
            }
            else
            {
                return 0;
            }
        } // END getRowCount()


        public Object getValueAt(int row, int col)
        {
            // First get the detail item "row"
            CashSaleDetail detail =
                (CashSaleDetail)currentSale.getDetailList().elementAt(row);
            ProductDesc product = detail.getProductDesc();

			// Lets table auto-format native data types
            Object data;

            // Now do the grunt work to map a column to an attribute
            // {"Item", "Name", "Unit", "Qty", "Price"};
            switch (col)
            {
                case 0: // Item
                    data = product.getItemNumber();
                    break;
                case 1: // Name
                    data = product.getName();
                    break;
                case 2: // Unit Price
                    data = currencyFormat.format(product.getPrice());
                    break;
                case 3: // Qty
                    data = new Integer(detail.getQty());
                    break;
                case 4: // Price
                    data = currencyFormat.format(product.calcPriceForQty(detail.getQty()));
                    break;
                default:
                    data = "Unknown!";
                    break;
            } // ENDSWITCH
            return data;
        } // END getValueAt()

        // The default implementations of these methods in
        // AbstractTableModel would work, but we can refine them.
        public String getColumnName(int column)
        {
            return colNames[column];
        }

        // Don't allow any editing
        public boolean isCellEditable(int row, int col)
        {
            return false;
        }

        public void setValueAt(Object aValue, int row, int column)
        {
            //
            // data[row][column] = aValue;
        }

        /** Update the table with the changed data model. */
        public void dataChanged(int changeType, int index)
        {
            if (changeType == ADDED_DETAIL)
            {
                // Paint the part of the table that changed
                fireTableDataChanged();
                //fireTableRowsInserted(index, index);
            }
            else
            {
                // Use this to redo the whole table
                // since all rows are effectively deleted!
                fireTableDataChanged();
            }
        } // END dataChanged()

        /**
         * JTable uses this method to determine the default renderer/ editor
         * for each cell to improve the display.
         */
        public Class getColumnClass(int c)
        {
            return getValueAt(0, c).getClass();
        }
    }; // ENDCLASS TableDataModel
//=================================================

} // ENDCLASS POSFrame

