/*-----------------------------------------------------------------------------------
Copyright (c)2003 Borland Software Corporation. Patents pending. All rights reserved.
-----------------------------------------------------------------------------------*/

package problem_domain;

import java.math.BigDecimal;
import java.util.Date;

/**
 * This is the class used to hold the price of the product. The intent is to provide a ProductDesc object with the
 * ability to hold more than just a fixed price. With the power of this added class, one can have prices that are
 * valid over different time intervals. <p> This class is based on Peter Coad's Components from
 * <i>Java Modeling in Color with UML</i> <p>
 * @stereotype moment-interval
 * @version 1.0 07/31/99
 * @see ProductDesc
 * @persistent*/
public class ProductPrice {

/* ========================================
 * Constructors
 * ======================================== */

    public ProductPrice(Date start, Date end, BigDecimal aPrice) {
        // Assert that end > start
        setValidStarting(start);
        setNotValidAfter(end);
        setPrice(aPrice);
        setPriceQty(1);
        setPriceUOM(1);
        setStatus(ProductPrice.PRICE_VALID_STATUS);
    }

/* ========================================
 * Business Methods
 * ======================================== */

    /**
     * Used as the <b>start</b> of the applicable interval for determining if price is valid.
     * @see +isValidOnDate()
     * @rdbLogicalType DATE
     */
    private Date validStarting;

    /**
     * Used as the <b>end</b> of the applicable interval for determining if price is valid.
     * @see +isValidOnDate()
     * @rdbLogicalType DATE
     */
    private Date notValidAfter;
    //    private int applicableInterval;

    /** This is the meat of this class...the <b>price</b> 
     * @rdbLogicalType NUMERIC*/
    private BigDecimal price;

    /**
     * The quantity that this price is valid for. For example: 1, 5, 10 <p> This could be used as a quantity discount
     * differentiator if pricing algorithm accounted for the possible existence of more than one
     * price per valid interval. <b>However,</b> it would be better to model this "feature" explicitly!
     * @rdbLogicalType INTEGER
     */
    private int priceQty = 1;

    /**
     * The unit of measure that this price is for. For example: each, dozen, gross <p>
     * In reality, this is generally a completely separate class that would include things like
     * conversion factors between UOMs.
     * @rdbLogicalType INTEGER
     */
    private int priceUOM = 1;

    /** Status of price 
     * @rdbLogicalType INTEGER*/
    private int status = ProductPrice.PRICE_INVALID_STATUS;

    /** Public constant to define a valid price status */
    public final static int PRICE_VALID_STATUS = 1;

    /** Public constant to define an <b>in</b>valid price status */
    public final static int PRICE_INVALID_STATUS = 2;

    public void makeProductPrice() {
    }

    /**  */
    public BigDecimal calcPriceForQty(int q) {
        return price.multiply(new BigDecimal(q));
    }

    /**
     * Check if this price is valid on the passed-in date. <p> Price is valid if date being checked is
     * greater than or equal to Starting and <b>not</b> greater than ValidAfter date. <p>
     * @return True if date is within applicable interval else false.
     */
    public boolean isValidOnDate(Date date) {
        return false;
    }

/* ========================================
 * Accessor/Mutator Methods
 * ======================================== */

    public Date getValidStarting() {
        return validStarting;
    }

    public void setValidStarting(Date aValidStarting) {
        validStarting = aValidStarting;
    }

    public Date getNotValidAfter() {
        return notValidAfter;
    }

    public void setNotValidAfter(Date nva) {
        notValidAfter = nva;
    }

    public BigDecimal getPrice() {
        return price;
    }

    public void setPrice(BigDecimal aPrice) {
        price = aPrice;
    }

    public int getPriceQty() {
        return priceQty;
    }

    public void setPriceQty(int aPriceQty) {
        priceQty = aPriceQty;
    }

    public int getPriceUOM() {
        return priceUOM;
    }

    public void setPriceUOM(int aPriceUOM) {
        priceUOM = aPriceUOM;
    }

    public int getStatus() {
        return status;
    }

    public void setStatus(int aStatus) {
        status = aStatus;
    }
} // ENDCLASS ProductPrice

